import type { TMIEmoteMap, TMIEmoteSet } from '../models';

/* eslint-disable sort-keys-fix/sort-keys-fix */
/**
 * A fixed number of global smileys have tokens that contain HTML entities or regular expressions.
 * Naively, this means each token in an outgoing message must be escaped and run against multiple
 * regexes. To avoid this and maintain constant time lookups, we map these special tokens to their
 * permutations.
 *
 * The only emotes that use these special tokens are our global smileys (contained in emote set 0)
 * and the premium smileys included with Turbo/Prime (sets 33 and 42).
 *
 * TODO(mikeche): Look into merging this logic with Twilight's. Currently we generate token maps in
 * both places.
 */
const SMILEY_MAP: { [key: string]: string[] | undefined } = {
  // Map tokens that contain HTML entities to their unescaped representations.
  '\\:\\&gt\\;': [':>'],
  '\\&gt\\;\\(': ['>('],
  '\\&lt\\;\\]': ['<]'],
  '\\&lt\\;3': ['<3'],

  // Map tokens that contain optional characters and character classes to all possible matches.
  ':-?(?:7|L)': [':-7', ':-L', ':7', ':L'],
  '[oO](_|\\.)[oO]': ['o_o', 'O_O', 'o_O', 'O_o', 'o.o', 'O.O', 'o.O', 'O.o'],
  '\\;-?(p|P)': [';-p', ';-P', ';p', ';P'],
  '\\;-?\\)': [';-)', ';)'],
  '\\:-?(o|O)': [':-o', ':-O', ':o', ':O'],
  '\\:-?(p|P)': [':-p', ':-P', ':p', ':P'],
  '\\:-?(S|s)': [':-S', ':-s', ':s', ':S'],
  '\\:-?[\\\\/]': [':-/', ':-\\', ':/', ':\\'],
  '\\:-?[z|Z|\\|]': [':-|', ':-Z', ':-z', ':|', ':Z', ':z'],
  '\\:-?\\(': [':-(', ':('],
  '\\:-?\\)': [':-)', ':)'],
  '\\:-?D': [':-D', ':D'],
  '#-?[\\\\/]': ['#-/', '#-\\', '#/', '#\\'],
  'B-?\\)': ['B-)', 'B)'],
  'R-?\\)': ['R-)', 'R)'],
};
/* eslint-enable sort-keys-fix/sort-keys-fix */

export function createEmoteMap(emoteSets: TMIEmoteSet[]): TMIEmoteMap {
  const emoteMap: TMIEmoteMap = {};
  // Iterate through each set.
  for (const emoteSet of emoteSets) {
    // Iterate through each emote in a set.
    for (const emote of emoteSet.emotes) {
      // Iterate through each alias of the emote.
      const aliases = SMILEY_MAP[emote.token] || [emote.token];
      for (const alias of aliases) {
        emoteMap[alias] = { ...emote };
      }
    }
  }
  return emoteMap;
}
