import type { TMIMessageTags, TMIRoomState } from '../models';
import { createDefaultRoomState, updateRoomStateWithTags } from './room-state';

describe('createDefaultRoomState', () => {
  it('generates the default room state correctly', () => {
    expect(createDefaultRoomState()).toEqual({
      broadcasterLang: null,
      emoteOnly: false,
      followersOnly: false,
      mercury: false,
      r9k: false,
      slowMode: false,
      subsOnly: false,
    });
  });
});

describe('updateRoomStateWithTags', () => {
  interface TestOptions {
    expectedStateValues: Partial<TMIRoomState>;
    initialStateOverrides?: Partial<TMIRoomState>;
    tags: TMIMessageTags;
  }

  /** Creates an initial room state with optional overrides */
  const createInitialState = (
    overrides: Partial<TMIRoomState> = {},
  ): TMIRoomState => ({
    broadcasterLang: 'en',
    emoteOnly: false,
    followersOnly: false,
    mercury: false,
    r9k: false,
    slowMode: false,
    subsOnly: false,
    ...overrides,
  });

  /**
   * Creates an initial room state, updates it with the provided message tags,
   * and asserts that the new state matches the initial state + specific values.
   */
  const expectRoomState = (options: TestOptions) => {
    const initialState = createInitialState(options.initialStateOverrides);
    expect(updateRoomStateWithTags(initialState, options.tags)).toEqual({
      ...initialState,
      ...options.expectedStateValues,
    });
  };

  it('returns the same room state when no tags are provided', () => {
    expectRoomState({ expectedStateValues: {}, tags: {} });
  });

  describe('broadcasterLang', () => {
    it('gets updated correctly', () => {
      expectRoomState({
        expectedStateValues: { broadcasterLang: 'de' },
        initialStateOverrides: { broadcasterLang: 'en' },
        tags: { 'broadcaster-lang': 'de' },
      });
    });
  });

  describe('emoteOnly', () => {
    it('gets enabled correctly', () => {
      expectRoomState({
        expectedStateValues: { emoteOnly: true },
        initialStateOverrides: { emoteOnly: false },
        tags: { 'emote-only': 'true' },
      });
    });

    it('gets disabled correctly', () => {
      expectRoomState({
        expectedStateValues: { emoteOnly: false },
        initialStateOverrides: { emoteOnly: true },
        tags: { 'emote-only': 'false' },
      });
    });
  });

  describe('followersOnly and followersOnlyRequirement', () => {
    it('get enabled correctly', () => {
      expectRoomState({
        expectedStateValues: {
          followersOnly: true,
          followersOnlyRequirement: 5,
        },
        initialStateOverrides: {
          followersOnly: false,
          followersOnlyRequirement: undefined,
        },
        tags: { 'followers-only': '5' },
      });
    });

    it('get disabled correctly', () => {
      expectRoomState({
        expectedStateValues: {
          followersOnly: false,
          followersOnlyRequirement: undefined,
        },
        initialStateOverrides: {
          followersOnly: true,
          followersOnlyRequirement: 20,
        },
        tags: { 'followers-only': '-1' },
      });
    });

    it('change followersOnlyRequirement correctly,', () => {
      expectRoomState({
        expectedStateValues: {
          followersOnly: true,
          followersOnlyRequirement: 100,
        },
        initialStateOverrides: {
          followersOnly: true,
          followersOnlyRequirement: 20,
        },
        tags: { 'followers-only': '100' },
      });
    });
  });

  describe('mercury', () => {
    it('gets enabled correctly', () => {
      expectRoomState({
        expectedStateValues: { mercury: true },
        initialStateOverrides: { mercury: false },
        tags: { mercury: 'true' },
      });
    });

    it('gets disabled correctly', () => {
      expectRoomState({
        expectedStateValues: { mercury: false },
        initialStateOverrides: { mercury: true },
        tags: { mercury: 'false' },
      });
    });
  });

  describe('r9k', () => {
    it('gets enabled correctly', () => {
      expectRoomState({
        expectedStateValues: { r9k: true },
        initialStateOverrides: { r9k: false },
        tags: { r9k: 'true' },
      });
    });

    it('gets disabled correctly', () => {
      expectRoomState({
        expectedStateValues: { r9k: false },
        initialStateOverrides: { r9k: true },
        tags: { r9k: 'false' },
      });
    });
  });

  describe('slowMode and slowModeDuration', () => {
    it('get enabled correctly', () => {
      expectRoomState({
        expectedStateValues: { slowMode: true, slowModeDuration: 10 },
        initialStateOverrides: { slowMode: false, slowModeDuration: undefined },
        tags: { slow: '10' },
      });
    });

    it('get disabled correctly', () => {
      expectRoomState({
        expectedStateValues: { slowMode: false, slowModeDuration: undefined },
        initialStateOverrides: { slowMode: true, slowModeDuration: 30 },
        tags: { slow: '-1' },
      });
    });

    it('change duration correctly', () => {
      expectRoomState({
        expectedStateValues: { slowMode: true, slowModeDuration: 100 },
        initialStateOverrides: { slowMode: true, slowModeDuration: 50 },
        tags: { slow: '100' },
      });
    });
  });

  describe('subsOnly', () => {
    it('gets enabled correctly', () => {
      expectRoomState({
        expectedStateValues: { subsOnly: true },
        initialStateOverrides: { subsOnly: false },
        tags: { 'subs-only': 'true' },
      });
    });

    it('gets disabled correctly', () => {
      expectRoomState({
        expectedStateValues: { subsOnly: false },
        initialStateOverrides: { subsOnly: true },
        tags: { 'subs-only': 'false' },
      });
    });
  });
});
