import type { TMIMessageTags, TMIRoomState } from '../models';
import { Utils } from './utils';

export function createDefaultRoomState(): TMIRoomState {
  return {
    broadcasterLang: null,
    emoteOnly: false,
    followersOnly: false,
    mercury: false,
    r9k: false,
    slowMode: false,
    subsOnly: false,
  };
}

/**
 * Returns a new TMIRoomState object which reflects the roomState param updated with any relevant
 * tags.
 */
export function updateRoomStateWithTags(
  roomState: TMIRoomState,
  tags: TMIMessageTags,
): TMIRoomState {
  const updatedRoomState: Partial<TMIRoomState> = {
    ...parseBroadcasterLanguageTag(tags['broadcaster-lang']),
    ...parseEmoteOnlyTag(tags['emote-only']),
    ...parseFollowersOnlyTag(tags['followers-only']),
    ...parseMercuryTag(tags.mercury),
    ...parseR9KTag(tags.r9k),
    ...parseSlowModeTag(tags.slow),
    ...parseSubsOnlyTag(tags['subs-only']),
  };

  return {
    ...roomState,
    ...updatedRoomState,
  };
}

/**
 * Returns either an empty object, or an object with a broadcasterLang key.
 * The tag is undefined if the broadcaster language was not included as part of this room state
 * update, the empty string if the broadcaster explicitly specified no language requirement, or the
 * language string.
 */
function parseBroadcasterLanguageTag(tag?: string): {
  broadcasterLang?: string | null;
} {
  if (tag === undefined) {
    return {};
  }

  return {
    broadcasterLang: tag === '' ? null : tag,
  };
}

/**
 * Returns either an empty object, or an object with an emoteOnly key.
 * The tag is undefined if emote-only mode was not included as part of this room state update, or a
 * stringified boolean.
 */
function parseEmoteOnlyTag(tag?: string): { emoteOnly?: boolean } {
  return tag === undefined ? {} : { emoteOnly: Utils.parseBool(tag, false) };
}

/**
 * Returns either an empty object, or an object with the followersOnly and followersOnlyRequirement
 * keys.
 * The tag is undefined if followers-only mode was not included as part of this room state update,
 * or a stringified integer. The integer is -1 if followers-only mode is off, or the number of
 * minutes a user must have followed for in order to chat.
 */
function parseFollowersOnlyTag(tag?: string):
  | {
      followersOnly: boolean;
      followersOnlyRequirement: number | undefined;
    }
  | {} {
  if (tag === undefined) {
    return {};
  }

  const requirement = Utils.parseInt(tag, 0);
  return {
    followersOnly: requirement !== -1,
    followersOnlyRequirement: requirement !== -1 ? requirement : undefined,
  };
}

/**
 * Returns either an empty object, or an object with a mercury key.
 * The tag is undefined if mercury mode was not included as part of this room state update, or a
 * stringified boolean.
 */
function parseMercuryTag(tag?: string): { mercury?: boolean } {
  return tag === undefined ? {} : { mercury: Utils.parseBool(tag, false) };
}

/**
 * Returns either an empty object, or an object with an r9k key.
 * The tag is undefined if r9k mode was not included as part of this room state update, or a
 * stringified boolean.
 */
function parseR9KTag(tag?: string): { r9k?: boolean } {
  return tag === undefined ? {} : { r9k: Utils.parseBool(tag, false) };
}

/**
 * Returns either an empty object, or an object with the slowMode and slowModeDuration keys.
 * The tag is undefined if slow mode was not included as part of this room state update, or a
 * stringified integer. The integer is 0 if slow mode is off, or the number of seconds a user must
 * wait between sending messages.
 */
function parseSlowModeTag(tag?: string):
  | {
      slowMode: boolean;
      slowModeDuration: number | undefined;
    }
  | {} {
  if (tag === undefined) {
    return {};
  }

  const duration = Utils.parseInt(tag, 0);
  return {
    slowMode: duration > 0,
    slowModeDuration: duration > 0 ? duration : undefined,
  };
}

/**
 * Returns either an empty object, or an object with a subsOnly key.
 * The tag is undefined if subs-only mode was not included as part of this room state update, or a
 * stringified boolean.
 */
function parseSubsOnlyTag(tag?: string): { subsOnly?: boolean } {
  return tag === undefined ? {} : { subsOnly: Utils.parseBool(tag, false) };
}
