import { Utils } from './utils';

describe('Utils', () => {
  describe('coalesce', () => {
    it('returns the first value when it is defined', () => {
      const value = Utils.coalesce('test', 'helloworld');
      expect(value).toBe('test');
    });

    it('returns the second value when the first is undefined', () => {
      const value = Utils.coalesce(undefined, 'test');
      expect(value).toBe('test');
    });

    it('returns undefined when both values are undefined', () => {
      const value = Utils.coalesce(undefined, undefined);
      expect(value).toBeUndefined();
    });
  });

  describe('parseBool', () => {
    it('returns true when the string value is "1" or "true", false otherwise', () => {
      expect(Utils.parseBool('1', null)).toBe(true);
      expect(Utils.parseBool('true', null)).toBe(true);

      expect(Utils.parseBool('', null)).toBe(false);
      expect(Utils.parseBool('2', null)).toBe(false);
      expect(Utils.parseBool('truth', null)).toBe(false);
      expect(Utils.parseBool('false', null)).toBe(false);
    });

    it('returns default value when undefined', () => {
      expect(Utils.parseBool(undefined, false)).toBe(false);
      expect(Utils.parseBool(undefined, null)).toBeNull();
      expect(Utils.parseBool(undefined, undefined)).toBeUndefined();
    });
  });

  describe('parseInt', () => {
    it('returns an integer when the string value can be parsed to a number', () => {
      expect(Utils.parseInt('1', null)).toBe(1);
      expect(Utils.parseInt('0', null)).toBe(0);
      expect(Utils.parseInt('-455', null)).toBe(-455);
      // float numbers are converted to integers
      expect(Utils.parseInt('15.6', null)).toBe(15);
    });

    it('returns default value when NaN', () => {
      expect(Utils.parseInt('invalid', 420)).toBe(420);
      expect(Utils.parseInt(undefined, 420)).toBe(420);
      expect(Utils.parseInt(undefined, 0)).toBe(0);
      expect(Utils.parseInt('invalid', null)).toBeNull();
      expect(Utils.parseBool(undefined, null)).toBeNull();
      expect(Utils.parseInt('invalid', undefined)).toBeUndefined();
      expect(Utils.parseBool(undefined, undefined)).toBeUndefined();
    });
  });

  describe('isJustinfan', () => {
    it('returns true for a name that starts with "justinfan" and contains 1 or more digits after', () => {
      expect(Utils.isJustinfan('justinfan1')).toBeTruthy();
      expect(Utils.isJustinfan('justinfan128942378947234')).toBeTruthy();
      expect(Utils.isJustinfan('justinfan')).toBeFalsy();
      expect(Utils.isJustinfan('justinfan121221 ')).toBeFalsy();
      expect(Utils.isJustinfan('test')).toBeFalsy();
      expect(Utils.isJustinfan(undefined)).toBeFalsy();
    });
  });

  describe('randomGuestUsername', () => {
    it('returns a username that evaluates to true using isJustinfan', () => {
      expect(Utils.isJustinfan(Utils.randomGuestUsername())).toBeTruthy();
    });
  });

  describe('replaceAll', () => {
    const defaultText = 'Hello World this is a test Hello hello :D';

    it('replaces nothing with an empty replacement object', () => {
      const replacement = Utils.replaceAll(defaultText, {});
      expect(replacement).toBe(defaultText);
    });

    it('replaces nothing with a replacement object that contains no matching substrings', () => {
      const replacement = Utils.replaceAll(defaultText, { wow: 'WOW' });
      expect(replacement).toBe(defaultText);
    });

    it('replaces all occurences of a matching substring', () => {
      const replacement = Utils.replaceAll(defaultText, {
        Hello: 'HELLO',
        World: 'WORLD',
      });
      expect(replacement).toBe('HELLO WORLD this is a test HELLO hello :D');
    });
  });

  describe('unescapeHtml', () => {
    it('replaces escaped HTML entities with unescaped characters', () => {
      expect(Utils.unescapeHtml('\\&amp\\;\\&amp\\;')).toBe('&&');
      expect(Utils.unescapeHtml('\\&lt\\;\\&lt\\;')).toBe('<<');
      expect(Utils.unescapeHtml('\\&gt\\;\\&gt\\;')).toBe('>>');
      expect(Utils.unescapeHtml('\\&quot\\;\\&quot\\;')).toBe('""');
      expect(Utils.unescapeHtml('\\&#039\\;\\&#039\\;')).toBe(`''`);
      expect(
        Utils.unescapeHtml('\\&amp\\;\\&lt\\;\\&gt\\;\\&quot\\;\\&#039\\;'),
      ).toBe(`&<>"'`);
    });
  });

  describe('channel', () => {
    it('returns a string always prepended by #', () => {
      expect(Utils.channel(undefined)).toBe('#');
      expect(Utils.channel('')).toBe('#');
      expect(Utils.channel('channel')).toBe('#channel');
      expect(Utils.channel('#channel')).toBe('#channel');
    });
  });

  describe('username', () => {
    it('returns a blank string as a default value', () => {
      expect(Utils.username(undefined)).toBe('');
    });

    it('returns a string and removes a prepended # if necessary', () => {
      expect(Utils.username('user')).toBe('user');
      expect(Utils.username('#user')).toBe('user');
    });

    it('forces value to be lower case', () => {
      expect(Utils.username('UsEr')).toBe('user');
    });
  });

  describe('decodeTag', () => {
    it('replaces instances of IRC escaped characters', () => {
      expect(Utils.decodeTag('\\stest\\s')).toBe(' test ');
      expect(Utils.decodeTag('\\:test\\:')).toBe(';test;');
      expect(Utils.decodeTag('\\\\test\\\\')).toBe('\\test\\');
      expect(Utils.decodeTag('\\rtest\\r')).toBe('\\rtest\\r');
      expect(Utils.decodeTag('\\ntest\\n')).toBe('\\ntest\\n');
      expect(Utils.decodeTag('⸝test⸝⸝')).toBe(',test,,');
    });
  });
});
