export class Utils {
  // Return the second value if the first value is undefined.
  public static coalesce<T>(obj1: T, obj2: T): T {
    return obj1 === undefined ? obj2 : obj1;
  }

  /**
   * Parse a string as a boolean.
   * If the value is undefined, returns the defaultValue.
   * If the value is a string, returns true if the string is 'true' or '1'.
   */
  public static parseBool<T>(
    value: string | undefined,
    defaultValue: T,
  ): T | boolean {
    return value === undefined
      ? defaultValue
      : value === 'true' || value === '1';
  }

  /**
   * Parse string as an an integer.
   * If the value is undefined or an invalid number, returns the defaultValue.
   */
  public static parseInt<T>(value = '', defaultValue: T): T | number {
    const num = parseInt(value, 10);
    return !isNaN(num) ? num : defaultValue;
  }

  public static isError(value: any | undefined): value is Error {
    return value && (value as Error).message !== undefined;
  }

  // Username is a justinfan username.
  public static isJustinfan(username = ''): boolean {
    return RegExp('^(justinfan)(\\d+$)', 'g').test(username);
  }

  // Return a random justinfan username.
  public static randomGuestUsername(): string {
    return `justinfan${Math.floor(Math.random() * 80000 + 1000)}`;
  }

  // Replace all occurences of a string using an object.
  public static replaceAll(
    str = '',
    replacements: Record<string, string>,
  ): string {
    for (const key of Object.keys(replacements)) {
      str = str.replace(new RegExp(key, 'g'), replacements[key]);
    }

    return str;
  }

  public static unescapeHtml(safe: string): string {
    return safe
      .replace(/\\&amp\\;/g, '&')
      .replace(/\\&lt\\;/g, '<')
      .replace(/\\&gt\\;/g, '>')
      .replace(/\\&quot\\;/g, '"')
      .replace(/\\&#039\\;/g, "'");
  }

  // Return a valid channel name.
  public static channel(channel = ''): string {
    return channel.charAt(0) === '#'
      ? channel.toLowerCase()
      : `#${channel.toLowerCase()}`;
  }

  // Return a valid username.
  public static username(username = ''): string {
    return username.charAt(0) === '#'
      ? username.substring(1).toLowerCase()
      : username.toLowerCase();
  }

  public static decodeTag(value = ''): string {
    return Utils.replaceAll(Utils.coalesce(value, ''), {
      '\\\\:': ';',
      '\\\\\\\\': '\\',
      '\\\\s': ' ',
      '\\n': '\n',
      '\\r': '\r',
      '⸝': ',',
    });
  }
}
