# react-branch

Provides react bindings for the
[branch-sdk](https://github.com/BranchMetrics/web-branch-deep-linking-attribution)
package, published by branch.io.

Features:

- Exposes the entire `branch-sdk` api via hooks.

## Setting Up

### Minimum setup

```tsx
import { BranchRoot } from 'react-branch';

const BRANCH_API_KEY = 'your-branch-api-key';

function App() {
  return <BranchRoot apiKey={BRANCH_API_KEY}>{/* app */}</BranchRoot>;
}
```

### Passing options to `branch.init()`

```tsx
import { BranchRoot, BranchInitOptions } from 'react-branch';

const BRANCH_API_KEY = 'your-branch-api-key';
const BRANCH_INIT_OPTIONS = {
  metadata: { device_id: 'user-device-id' },
  no_journeys: true,
};

function App() {
  return (
    <BranchRoot apiKey={BRANCH_API_KEY} initOptions={BRANCH_INIT_OPTIONS}>
      {/* app */}
    </BranchRoot>
  );
}
```

## Basic Use

This package provides custom hooks for the most common use-cases of branch.

### branch.link()

```tsx
import { useLink, DeepLinkData } from 'react-branch';

const linkData: DeepLinkData = {};

function DeepLink() {
  const [deepLink] = useLink(linkData);
  return <a href={deepLink}>This is a deep link.</a>;
}
```

> Note: if you notice `useLink()` firing multiple requests to create a branch
> link you likely have to memoize the input `linkData` object. (you can use
> `React.useMemo()` to achieve this)

### branch.data()

```tsx
import { useData } from 'react-branch';

function UpsellButton() {
  const [sessionData] = useData();
  return (
    <button>
        {sessionData?.hasApp ? "Open" : "Get"} the app.
    <button>
  );
}
```

### branch.deepview()

#### as a hook

```tsx
import { useDeepView, DeepLinkData } from 'react-branch';

const deepviewData: DeepLinkData = {};

function PageWithDeepview() {
  useDeepView(deepviewData);
  return (/* page implementation */);
}
```

#### as a component

```tsx
import { Deepview, DeepLinkData } from 'react-branch';

const deepviewData: DeepLinkData = {};

function PageWithDeepview() {
  return (
    <>
      <Deepview data={deepviewData} />
      {/* page implementation */}
    </>
  );
}
```
