import type { BranchError, InitOptions, SessionData } from 'branch-sdk';
import BranchSDK from 'branch-sdk';
import type { FC } from 'react';
import {
  createContext,
  useEffect,
  useLayoutEffect,
  useMemo,
  useState,
} from 'react';

// doing this to avoid the SSR warning for useLayoutEffect
// branch.init needs to be fired synchronously in order for subsequent branch calls to succeed (race condition otherwise, since useEffect may be batched / is not guaranteed to be sync)
const useEnhancedEffect =
  typeof window === 'undefined' ? useEffect : useLayoutEffect;

export type BranchContext = {
  branch: typeof BranchSDK;
  branchInitError: BranchError;
  branchInitSessionData: SessionData | null;
};

export const branchContext = createContext<BranchContext>({
  branch: BranchSDK,
  branchInitError: null,
  branchInitSessionData: null,
});

export type BranchRootProps = {
  /**
   * Your branch.io api key.
   */
  apiKey: string;
  /**
   * Any options you want to pass to {@link https://help.branch.io/developers-hub/docs/web-full-reference#section-init-branch-key-options-callback branch.init()}.
   */
  initOptions?: InitOptions;
};

/**
 * The react-branch context provider. Wrap this around the root of your application.
 * It will initialize branch for you with the `apiKey` and `initOptions` you pass as props.
 *
 * @param {string} apiKey Your {@link https://branch.io branch.io} api key.
 * @param {InitOptions} initOptions Any options you want to pass to {@link https://help.branch.io/developers-hub/docs/web-full-reference#section-init-branch-key-options-callback branch.init()}.
 */
export const BranchRoot: FC<BranchRootProps> = ({
  apiKey,
  children,
  initOptions,
}) => {
  const [branchInitError, setBranchInitError] = useState<BranchError>(null);
  const [branchInitSessionData, setBranchInitSessionData] =
    useState<SessionData | null>(null);

  // useEffect on the server, useLayoutEffect on the client
  // neither actually runs on the server, but need it for the call signature to be stable
  useEnhancedEffect(() => {
    let mounted = true;
    BranchSDK.init(apiKey, initOptions, (err, sessionData) => {
      if (!mounted) {
        return;
      }
      if (err) {
        return setBranchInitError(err);
      }
      setBranchInitSessionData(sessionData);
    });
    return () => {
      mounted = false;
    };
    // empty dep array because we only ever want to do this once on mount
  }, []);

  const ctx = useMemo(
    () => ({
      branch: BranchSDK,
      branchInitError,
      branchInitSessionData,
    }),
    [branchInitError, branchInitSessionData],
  );

  return <branchContext.Provider children={children} value={ctx} />;
};
BranchRoot.displayName = 'BranchRoot';
