import { act } from '@testing-library/react-hooks';
import type { SessionData } from 'branch-sdk';
import branchSDK from 'branch-sdk';
import type { FC } from 'react';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { generateInitOptions, generateSessionData } from '../test-utils';
import { useBranch } from '../useBranch';
import { BranchRoot } from '.';

jest.mock('branch-sdk');

describe(BranchRoot, () => {
  afterEach(() => {
    jest.resetAllMocks();
  });

  const ChildTargetElement: FC<{
    error?: string | null;
    session: SessionData | null;
  }> = () => <div />;
  const TargetElementWrapper: FC = () => {
    const { branchInitError, branchInitSessionData } = useBranch();
    return (
      <ChildTargetElement
        error={branchInitError}
        session={branchInitSessionData}
      />
    );
  };

  const setup = createMountWrapperFactory(BranchRoot, () => ({
    apiKey: 'test-api-key',
    children: <TargetElementWrapper />,
  }));

  it('renders child element as expected', () => {
    const { wrapper } = setup();

    expect(wrapper.find(ChildTargetElement)).toExist();
  });

  it('does never call branch.init() multiple times', () => {
    const { wrapper } = setup();

    expect(wrapper.find(ChildTargetElement)).toExist();
    expect(branchSDK.init).toHaveBeenCalledTimes(1);

    wrapper.setProps({ apiKey: 'different-key' });
    wrapper.update();
    expect(branchSDK.init).toHaveBeenCalledTimes(1);
  });

  describe('success cases', () => {
    const successData = generateSessionData();
    const successMockImplementation = (_: any, __: any, cb: any) =>
      cb(null, successData);

    it('calls  branch.init() on mount', () => {
      (
        branchSDK.init as jest.MockedFunction<typeof branchSDK.init>
      ).mockImplementation(successMockImplementation);

      const { props, wrapper } = setup();

      expect(wrapper.find(ChildTargetElement)).toExist();
      expect(branchSDK.init).toHaveBeenCalledTimes(1);
      expect(branchSDK.init).toHaveBeenCalledWith(
        props.apiKey,
        props.initOptions,
        expect.any(Function),
      );
      expect(wrapper.find(ChildTargetElement)).toHaveProp({
        session: successData,
      });
    });

    it('calls  branch.init() on mount with initOptions', () => {
      (
        branchSDK.init as jest.MockedFunction<typeof branchSDK.init>
      ).mockImplementation(successMockImplementation);

      const { props, wrapper } = setup({ initOptions: generateInitOptions() });

      expect(wrapper.find(ChildTargetElement)).toExist();
      expect(branchSDK.init).toHaveBeenCalledTimes(1);
      expect(branchSDK.init).toHaveBeenCalledWith(
        props.apiKey,
        props.initOptions,
        expect.any(Function),
      );
      expect(wrapper.find(ChildTargetElement)).toHaveProp({
        session: successData,
      });
    });
  });

  describe('error cases', () => {
    const errorMessage = 'error';
    const failureMockImplementation = (_: any, __: any, cb: any) =>
      cb(errorMessage, null);

    it('calls  branch.init() on mount', () => {
      (
        branchSDK.init as jest.MockedFunction<typeof branchSDK.init>
      ).mockImplementation(failureMockImplementation);

      const { props, wrapper } = setup();

      expect(wrapper.find(ChildTargetElement)).toExist();
      expect(branchSDK.init).toHaveBeenCalledTimes(1);
      expect(branchSDK.init).toHaveBeenCalledWith(
        props.apiKey,
        props.initOptions,
        expect.any(Function),
      );
      expect(wrapper.find(ChildTargetElement)).toHaveProp({
        error: errorMessage,
      });
    });

    it('calls  branch.init() on mount with initOptions', () => {
      (
        branchSDK.init as jest.MockedFunction<typeof branchSDK.init>
      ).mockImplementation(failureMockImplementation);

      const { props, wrapper } = setup({
        initOptions: generateInitOptions(),
      });

      expect(wrapper.find(ChildTargetElement)).toExist();
      expect(branchSDK.init).toHaveBeenCalledTimes(1);
      expect(branchSDK.init).toHaveBeenCalledWith(
        props.apiKey,
        props.initOptions,
        expect.any(Function),
      );
      expect(wrapper.find(ChildTargetElement)).toHaveProp({
        error: errorMessage,
      });
    });
  });

  describe('unmount before callback cases', () => {
    let doCallback = () => null;
    const successData = generateSessionData();
    const successMockImplementation = (_: any, __: any, cb: any) => {
      doCallback = () => cb(null, successData);
    };

    it('does not error if branch.init() callback returns after unmounting', () => {
      (
        branchSDK.init as jest.MockedFunction<typeof branchSDK.init>
      ).mockImplementation(successMockImplementation);

      const { props, wrapper } = setup();

      expect(wrapper.find(ChildTargetElement)).toExist();
      expect(branchSDK.init).toHaveBeenCalledTimes(1);
      expect(branchSDK.init).toHaveBeenCalledWith(
        props.apiKey,
        props.initOptions,
        expect.any(Function),
      );
      expect(wrapper.find(ChildTargetElement)).toHaveProp({
        session: null,
      });

      wrapper.unmount();
      act(() => {
        doCallback();
      });
    });
  });
});
