import type { BranchError, SessionData } from 'branch-sdk';
import { useEffect, useState } from 'react';
import { useBranch } from '../useBranch';

/**
 * Hook that utilizes the {@link https://help.branch.io/developers-hub/docs/web-full-reference#section-data-callback branch.data()} and {@link https://help.branch.io/developers-hub/docs/web-full-reference#section-first-callback branch.first()} to provide access to user's branch session data.
 *
 * @param {boolean} [first=false] whether to use `branch.first()` instead of `branch.data()`. Default: false
 * @returns {([SessionData | null, BranchError])} array of [SessionData, PossibleError]
 */
export function useData(first = false): [SessionData | null, BranchError] {
  const [error, setError] = useState<BranchError>(null);
  const [data, setData] = useState<SessionData | null>(null);
  const { branch } = useBranch();

  useEffect(() => {
    let mounted = true;
    if (first) {
      branch.first((err, sessionData) => {
        if (!mounted) {
          return;
        }
        if (err) {
          return setError(err);
        }
        setData(sessionData);
      });
    } else {
      branch.data((err, sessionData) => {
        if (!mounted) {
          return;
        }
        if (err) {
          return setError(err);
        }
        setData(sessionData);
      });
    }

    return () => {
      mounted = false;
    };
  }, [branch, first]);

  return [data, error];
}
