import { act, renderHook } from '@testing-library/react-hooks';
import branchSDK from 'branch-sdk';
import { generateSessionData } from '../test-utils';
import { useData } from '.';

jest.mock('branch-sdk');

describe(useData, () => {
  afterEach(() => {
    jest.resetAllMocks();
  });

  describe('successful cases', () => {
    const successData = generateSessionData();
    const successMockImplementation = (cb: any) => cb(null, successData);

    it('calls branch.data() when first is false', () => {
      (
        branchSDK.data as jest.MockedFunction<typeof branchSDK.data>
      ).mockImplementation(successMockImplementation);
      const { result } = renderHook(() => useData());

      expect(result.current).toStrictEqual([successData, null]);
      expect(branchSDK.data).toHaveBeenCalled();
      expect(branchSDK.first).not.toHaveBeenCalled();
    });

    it('calls branch.first() when first is true', () => {
      (
        branchSDK.first as jest.MockedFunction<typeof branchSDK.first>
      ).mockImplementation(successMockImplementation);
      const { result } = renderHook(() => useData(true));

      expect(result.current).toStrictEqual([successData, null]);
      expect(branchSDK.data).not.toHaveBeenCalled();
      expect(branchSDK.first).toHaveBeenCalled();
    });
  });

  describe('error cases', () => {
    const errorMessage = 'error';
    const failureMockImplementation = (cb: any) => cb(errorMessage, null);

    it('calls branch.data() when first is false', () => {
      (
        branchSDK.data as jest.MockedFunction<typeof branchSDK.data>
      ).mockImplementation(failureMockImplementation);
      const { result } = renderHook(() => useData());

      expect(result.current).toStrictEqual([null, errorMessage]);
      expect(branchSDK.data).toHaveBeenCalled();
      expect(branchSDK.first).not.toHaveBeenCalled();
    });

    it('calls branch.first() when first is true', () => {
      (
        branchSDK.first as jest.MockedFunction<typeof branchSDK.first>
      ).mockImplementation(failureMockImplementation);
      const { result } = renderHook(() => useData(true));

      expect(result.current).toStrictEqual([null, errorMessage]);
      expect(branchSDK.data).not.toHaveBeenCalled();
      expect(branchSDK.first).toHaveBeenCalled();
    });
  });

  describe('unmount before callback cases', () => {
    const successData = generateSessionData();
    let doCallback = () => null;
    const successMockImplementation = (cb: any) => {
      doCallback = () => cb(null, successData);
    };

    it('does not error if branch.data() callback returns after unmounting', () => {
      (
        branchSDK.data as jest.MockedFunction<typeof branchSDK.data>
      ).mockImplementation(successMockImplementation);
      expect(branchSDK.data).not.toHaveBeenCalled();
      const { result, unmount } = renderHook(() => useData());
      expect(branchSDK.data).toHaveBeenCalled();
      expect(result.current).toStrictEqual([null, null]);

      unmount();
      act(() => {
        doCallback();
      });

      expect(result.error).toBeUndefined();
    });

    it('does not error if branch.first() callback returns after unmounting', () => {
      (
        branchSDK.first as jest.MockedFunction<typeof branchSDK.first>
      ).mockImplementation(successMockImplementation);
      expect(branchSDK.first).not.toHaveBeenCalled();
      const { result, unmount } = renderHook(() => useData(true));
      expect(branchSDK.first).toHaveBeenCalled();
      expect(result.current).toStrictEqual([null, null]);

      unmount();
      act(() => {
        doCallback();
      });

      expect(result.error).toBeUndefined();
    });
  });
});
