import type { BranchError, DeepLinkData, DeepViewOptions } from 'branch-sdk';
import { useState } from 'react';
import useDeepCompareEffect from 'use-deep-compare-effect';
import { useBranch } from '../useBranch';

/**
 * Provides a way to configure a dynamic {@link https://help.branch.io/developers-hub/docs/web-full-reference#section-deepview-data-options-callback deepview}.
 *
 * @param {DeepLinkData} deepviewData The data to configure this deepview.
 * @param {DeepViewOptions} [options] The options for this deepview.
 * @returns {[BranchError]}
 */
export function useDeepview(
  deepviewData: DeepLinkData,
  options?: DeepViewOptions,
): [BranchError] {
  const { branch } = useBranch();
  const [error, setError] = useState<BranchError>(null);

  useDeepCompareEffect(() => {
    let mounted = true;
    // the new object here is necessary because branch mutates the deepviewData.data object when we call branch.deepview() which would otherwise cause the deepEffect to re-trigger
    const clonedDeepviewData = {
      ...deepviewData,
      data: { ...deepviewData.data },
    };
    // the new object here is necessary because branch mutates the options object when we call branch.deepview() which would otherwise cause the deepEffect to re-trigger
    const clonedDeepviewOptions = options && {
      ...options,
    };
    branch.deepview(clonedDeepviewData, clonedDeepviewOptions, (err) => {
      if (!mounted) {
        return;
      }
      if (err) {
        setError(err);
      }
    });

    return () => {
      mounted = false;
    };
  }, [branch, deepviewData, options]);

  return [error];
}
