import { act, renderHook } from '@testing-library/react-hooks';
import branchSDK from 'branch-sdk';
import { generateDeepLinkData, generateDeepViewOptions } from '../test-utils';
import { useDeepview } from '.';

jest.mock('branch-sdk');

describe(useDeepview, () => {
  const defaultDeepviewData = generateDeepLinkData();
  const defaultDeepviewOptions = generateDeepViewOptions();

  afterEach(() => {
    jest.resetAllMocks();
  });

  describe('successful cases', () => {
    const successMockImplementation = (_: any, __: any, cb: any) => cb(null);
    it('calls branch.deepview() with the expected deepview data', () => {
      (
        branchSDK.deepview as jest.MockedFunction<typeof branchSDK.deepview>
      ).mockImplementation(successMockImplementation);

      const { result } = renderHook(
        ({ deepviewData, deepviewOptions }) =>
          useDeepview(deepviewData, deepviewOptions),
        {
          initialProps: {
            deepviewData: defaultDeepviewData,
            deepviewOptions: undefined,
          },
        },
      );

      expect(result.current).toStrictEqual([null]);
      expect(branchSDK.deepview).toHaveBeenCalledWith(
        defaultDeepviewData,
        undefined,
        expect.any(Function),
      );
    });

    it('calls branch.deepview() with the expected deepview & option data', () => {
      (
        branchSDK.deepview as jest.MockedFunction<typeof branchSDK.deepview>
      ).mockImplementation(successMockImplementation);

      const { result } = renderHook(
        ({ deepviewData, deepviewOptions }) =>
          useDeepview(deepviewData, deepviewOptions),
        {
          initialProps: {
            deepviewData: defaultDeepviewData,
            deepviewOptions: defaultDeepviewOptions,
          },
        },
      );

      expect(result.current).toStrictEqual([null]);
      expect(branchSDK.deepview).toHaveBeenCalledWith(
        defaultDeepviewData,
        defaultDeepviewOptions,
        expect.any(Function),
      );
    });

    it('does not call branch.deepview() again if values have not changed (deep compare)', () => {
      (
        branchSDK.deepview as jest.MockedFunction<typeof branchSDK.deepview>
      ).mockImplementation(successMockImplementation);

      expect(branchSDK.deepview).toHaveBeenCalledTimes(0);

      const { rerender, result } = renderHook(
        ({ deepviewData, deepviewOptions }) =>
          useDeepview(deepviewData, deepviewOptions),
        {
          initialProps: {
            deepviewData: defaultDeepviewData,
            deepviewOptions: undefined,
          },
        },
      );

      expect(result.current).toStrictEqual([null]);
      expect(branchSDK.deepview).toHaveBeenCalledTimes(1);
      expect(branchSDK.deepview).toHaveBeenCalledWith(
        defaultDeepviewData,
        undefined,
        expect.any(Function),
      );

      // re-render  with the same data, but a new object
      rerender({
        deepviewData: { ...defaultDeepviewData },
        deepviewOptions: undefined,
      });

      expect(branchSDK.deepview).toHaveBeenCalledTimes(1);

      // change a nested key, we now re-render the effect
      const changedDeepViewProps = {
        deepviewData: {
          ...defaultDeepviewData,
          data: {
            ...defaultDeepviewData.data,
            customKey: 'has_changed',
          },
        },
        deepviewOptions: undefined,
      };
      rerender(changedDeepViewProps);

      expect(branchSDK.deepview).toHaveBeenCalledTimes(2);
      expect(branchSDK.deepview).toHaveBeenCalledWith(
        changedDeepViewProps.deepviewData,
        changedDeepViewProps.deepviewOptions,
        expect.any(Function),
      );
    });
  });

  describe('error cases', () => {
    const errorMessage = 'error';
    const failureMockImplementation = (_: any, __: any, cb: any) =>
      cb(errorMessage, null);

    it('calls branch.deepview() with the expected deepview data', () => {
      (
        branchSDK.deepview as jest.MockedFunction<typeof branchSDK.deepview>
      ).mockImplementation(failureMockImplementation);

      const { result } = renderHook(
        ({ deepviewData, deepviewOptions }) =>
          useDeepview(deepviewData, deepviewOptions),
        {
          initialProps: {
            deepviewData: defaultDeepviewData,
            deepviewOptions: undefined,
          },
        },
      );

      expect(result.current).toStrictEqual([errorMessage]);
      expect(branchSDK.deepview).toHaveBeenCalledWith(
        defaultDeepviewData,
        undefined,
        expect.any(Function),
      );
    });

    it('calls branch.deepview() with the expected deepview & option data', () => {
      (
        branchSDK.deepview as jest.MockedFunction<typeof branchSDK.deepview>
      ).mockImplementation(failureMockImplementation);

      const { result } = renderHook(
        ({ deepviewData, deepviewOptions }) =>
          useDeepview(deepviewData, deepviewOptions),
        {
          initialProps: {
            deepviewData: defaultDeepviewData,
            deepviewOptions: defaultDeepviewOptions,
          },
        },
      );

      expect(result.current).toStrictEqual([errorMessage]);
      expect(branchSDK.deepview).toHaveBeenCalledWith(
        defaultDeepviewData,
        defaultDeepviewOptions,
        expect.any(Function),
      );
    });
  });

  describe('unmount before callback cases', () => {
    const errorMessage = 'error';
    let doCallback = () => null;
    const failureMockImplementation = (_: any, __: any, cb: any) => {
      doCallback = () => cb(errorMessage, null);
    };

    it('does not error if branch.deepview() callback returns after unmounting', async () => {
      (
        branchSDK.deepview as jest.MockedFunction<typeof branchSDK.deepview>
      ).mockImplementation(failureMockImplementation);

      const { result, unmount } = renderHook(
        ({ deepviewData, deepviewOptions }) =>
          useDeepview(deepviewData, deepviewOptions),
        {
          initialProps: {
            deepviewData: defaultDeepviewData,
            deepviewOptions: undefined,
          },
        },
      );

      expect(branchSDK.deepview).toHaveBeenCalledWith(
        defaultDeepviewData,
        undefined,
        expect.any(Function),
      );
      expect(result.current).toStrictEqual([null]);

      unmount();

      act(() => {
        doCallback();
      });

      expect(result.error).toBeUndefined();
    });
  });
});
