import type { BranchError, DeepLinkData } from 'branch-sdk';
import { useState } from 'react';
import useDeepCompareEffect from 'use-deep-compare-effect';
import { useBranch } from '../useBranch';
/**
 * Generates a branch link based on the linkData using {@link https://help.branch.io/developers-hub/docs/web-full-reference#section-link-data-callback `branch.link()`}.
 *
 * @param {DeepLinkData} linkData Link data passed to branch for generating the deep link.
 * @returns {([string | null, BranchError])} array of `[url, potential error]`
 */
export function useLink(linkData: DeepLinkData): [string | null, BranchError] {
  const [linkError, setLinkError] = useState<string | null>(null);
  const [url, setUrl] = useState<string | null>(null);
  const { branch } = useBranch();

  useDeepCompareEffect(() => {
    let mounted = true;
    // the new object here is necessary because branch mutates the linkData.data object when we call branch.link() which would otherwise cause the deepEffect to re-trigger
    const clonedLinkData = {
      ...linkData,
      data: { ...linkData.data },
    };
    branch.link(clonedLinkData, (err, link) => {
      if (!mounted) {
        return;
      }
      if (err) {
        return setLinkError(err);
      }
      setUrl(link);
    });
    return () => {
      mounted = false;
    };
  }, [branch, linkData]);

  return [url, linkError];
}
