import { renderHook } from '@testing-library/react-hooks';
import branchSDK from 'branch-sdk';
import { generateDeepLinkData } from '../test-utils';
import { useLink } from '.';

jest.mock('branch-sdk');

describe(useLink, () => {
  const linkData = generateDeepLinkData();

  afterEach(() => {
    jest.resetAllMocks();
  });

  describe('success cases', () => {
    const mockLink = 'test.app.link';
    const successMockImplementation = (_: any, cb: any) => cb(null, mockLink);

    it('calls branch.link() with the expected link data', () => {
      (
        branchSDK.link as jest.MockedFunction<typeof branchSDK.link>
      ).mockImplementation(successMockImplementation);
      const { result } = renderHook(() => useLink(linkData));
      expect(result.current).toStrictEqual([mockLink, null]);
      expect(branchSDK.link).toHaveBeenCalledTimes(1);
      expect(branchSDK.link).toHaveBeenCalledWith(
        linkData,
        expect.any(Function),
      );
    });

    it('does not call branch.link() again if values have not changed (deep compare)', () => {
      (
        branchSDK.link as jest.MockedFunction<typeof branchSDK.link>
      ).mockImplementation(successMockImplementation);

      expect(branchSDK.link).toHaveBeenCalledTimes(0);

      const { rerender, result } = renderHook((props) => useLink(props), {
        initialProps: linkData,
      });

      expect(result.current).toStrictEqual([mockLink, null]);
      expect(branchSDK.link).toHaveBeenCalledTimes(1);
      expect(branchSDK.link).toHaveBeenCalledWith(
        linkData,
        expect.any(Function),
      );

      // re-render  with the same data, but a new object
      rerender({ ...linkData });

      expect(branchSDK.link).toHaveBeenCalledTimes(1);

      // change a nested key, we now re-render the effect
      const changedLinkData = {
        ...linkData,
        data: { ...linkData.data, customKey: 'has_changed' },
      };
      rerender(changedLinkData);

      expect(branchSDK.link).toHaveBeenCalledTimes(2);
      expect(branchSDK.link).toHaveBeenCalledWith(
        changedLinkData,
        expect.any(Function),
      );
    });
  });

  describe('error cases', () => {
    const errorMessage = 'error';
    const failureMockImplementation = (_: any, cb: any) =>
      cb(errorMessage, null);

    it('calls branch.link() with the expected link data', () => {
      (
        branchSDK.link as jest.MockedFunction<typeof branchSDK.link>
      ).mockImplementation(failureMockImplementation);
      const { result } = renderHook(() => useLink(linkData));

      expect(result.current).toStrictEqual([null, errorMessage]);
      expect(branchSDK.link).toHaveBeenCalledWith(
        linkData,
        expect.any(Function),
      );
    });
  });

  describe('unmount before callback cases', () => {
    const mockLink = 'test.app.link';
    let doCallback = () => null;
    const successMockImplementation = (_: any, cb: any) => {
      doCallback = () => cb(null, mockLink);
    };

    it('does not error if branch.link() callback returns after unmounting', () => {
      (
        branchSDK.link as jest.MockedFunction<typeof branchSDK.link>
      ).mockImplementation(successMockImplementation);
      const { result, unmount } = renderHook(() => useLink(linkData));
      expect(result.current).toStrictEqual([null, null]);
      expect(branchSDK.link).toHaveBeenCalledTimes(1);
      expect(branchSDK.link).toHaveBeenCalledWith(
        linkData,
        expect.any(Function),
      );

      unmount();
      doCallback();

      expect(result.error).toBeUndefined();
    });
  });
});
