import { GAMEPAD_AXES, GAMEPAD_BUTTONS } from '../../constants';
import { GamepadAxisEvent, GamepadButtonEvent } from '../../types';
import type { BlockingListeners, Listeners } from '../../types';
import { getEventName } from '../index';
import { fireEvent } from '.';

describe('fireEvent', () => {
  it('calls a blocking listener and does not call additional listeners when a blocking listener is registered', () => {
    const mockListener = jest.fn();
    const mockBlockingListener = jest.fn();
    const inputName = GAMEPAD_BUTTONS[0];

    const eventName = getEventName({
      eventType: GamepadButtonEvent.onDown,
      inputName,
    });
    const blockingListeners: BlockingListeners = {};
    blockingListeners[eventName] = mockBlockingListener;
    const listeners: Listeners = {};
    listeners[eventName] = new Set();
    listeners[eventName]?.add(mockListener);

    // fire event with blocking listener & listeners
    fireEvent({ blockingListeners, eventName, gamepadIndex: 0, listeners });

    expect(mockListener).not.toHaveBeenCalled();
    expect(mockBlockingListener).toHaveBeenCalledTimes(1);

    // remove blocking listener
    blockingListeners[eventName] = undefined;

    // fire event with only standard listeners
    fireEvent({ blockingListeners, eventName, gamepadIndex: 0, listeners });

    expect(mockListener).toHaveBeenCalledTimes(1);
    expect(mockBlockingListener).toHaveBeenCalledTimes(1);
  });

  it('calls all listeners when a blocking listener is not registered', () => {
    const mockListenerOne = jest.fn();
    const mockListenerTwo = jest.fn();
    const mockListenerThree = jest.fn();
    const inputName = GAMEPAD_BUTTONS[0];
    const eventName = getEventName({
      eventType: GamepadButtonEvent.onDown,
      inputName,
    });
    const blockingListeners: BlockingListeners = {};
    const listeners: Listeners = {};

    listeners[eventName] = new Set();
    // Add all three listeners
    listeners[eventName]?.add(mockListenerOne);
    listeners[eventName]?.add(mockListenerTwo);
    listeners[eventName]?.add(mockListenerThree);

    // fire event with all three listeners
    fireEvent({ blockingListeners, eventName, gamepadIndex: 0, listeners });

    expect(mockListenerOne).toHaveBeenCalledTimes(1);
    expect(mockListenerTwo).toHaveBeenCalledTimes(1);
    expect(mockListenerThree).toHaveBeenCalledTimes(1);

    // remove a listener
    listeners[eventName]?.delete(mockListenerTwo);

    // fire event with only remaining listeners
    fireEvent({ blockingListeners, eventName, gamepadIndex: 0, listeners });

    expect(mockListenerOne).toHaveBeenCalledTimes(2);
    expect(mockListenerTwo).toHaveBeenCalledTimes(1);
    expect(mockListenerThree).toHaveBeenCalledTimes(2);
  });

  it('passes the value to listeners when one exists', () => {
    const mockListenerOne = jest.fn();
    const mockListenerTwo = jest.fn();
    const AXIS_CHANGED_VALUE = 0.85;
    const axisEventName = getEventName({
      eventType: GamepadAxisEvent.onValueChanged,
      inputName: GAMEPAD_AXES[0],
    });
    const buttonEventName = getEventName({
      eventType: GamepadButtonEvent.onUp,
      inputName: GAMEPAD_BUTTONS[0],
    });
    const blockingListeners: BlockingListeners = {};
    const listeners: Listeners = {};

    listeners[axisEventName] = new Set();
    listeners[buttonEventName] = new Set();

    listeners[axisEventName]?.add(mockListenerOne);
    listeners[buttonEventName]?.add(mockListenerTwo);

    // fire both events
    fireEvent({
      blockingListeners,
      eventName: axisEventName,
      gamepadIndex: 0,
      listeners,
      value: AXIS_CHANGED_VALUE,
    });
    fireEvent({
      blockingListeners,
      eventName: buttonEventName,
      gamepadIndex: 0,
      listeners,
    });

    expect(mockListenerOne).toHaveBeenCalledWith(0, AXIS_CHANGED_VALUE);
    expect(mockListenerTwo).toHaveBeenCalledWith(0, undefined);
  });
});
