import type { FC } from 'react';
import { useCallback, useEffect, useMemo, useRef } from 'react';
import { Platform, useStaticEnvironment } from 'tachyon-environment';
import { logger } from 'tachyon-logger';
import { isBrowser } from 'tachyon-utils-stdlib';
import { GAMEPAD_AXES, GAMEPAD_BUTTONS } from '../constants';
import { GamepadAxisEvent, GamepadButtonEvent } from '../types';
import type {
  BlockingListeners,
  GamepadEventName,
  GamepadReducerActions,
  GamepadState,
  GetEventNameOpts,
  Listeners,
  SubscribeToInputOpts,
  UpdateGamepadReducerOpts,
} from '../types';
import type {
  SubscribeToGamepadAxisInputEventOpts,
  SubscribeToGamepadButtonInputEventOpts,
} from '../useGamepadInput';
import { gamepadInputContext } from '../useGamepadInput';
import { fireEvent } from './fireEvent';

const GamepadUnsupportedPlatform: Array<Platform> = [Platform.PS4];

// Reducer function that's used to keep our hook's state updated. We need to
// keep track of which buttons were pressed during the previous frame so
// that we can properly fire one time events for when a button is first pressed
// down or released up.
export function updateGamepad({
  blockingListeners,
  gamepads,
  listeners,
  oldState,
}: UpdateGamepadReducerOpts): Map<number, GamepadState> {
  let wasStateUpdated = false;
  const newState = new Map(oldState);
  for (let gamepadIndex = 0; gamepadIndex < gamepads.length; gamepadIndex++) {
    const gamepad = gamepads[gamepadIndex];
    const previousGamepadState = oldState.get(gamepadIndex);
    gamepad?.axes.forEach((value, axisIndex) => {
      if (value !== 0) {
        // fire on non-zero axis values
        fireEvent({
          blockingListeners,
          eventName: getEventName({
            eventType: GamepadAxisEvent.onValueChanged,
            inputName: GAMEPAD_AXES[axisIndex],
          }),
          gamepadIndex,
          listeners,
          value,
        });
      }
    });

    // Iterate over each button index to button pair in our gamepad mapping
    // and fire appropriate events
    gamepad?.buttons.forEach((button, buttonIndex) => {
      const isCurrentlyPressed = button.pressed;
      const wasPressedLastFrame =
        previousGamepadState?.pressedButtons[buttonIndex];
      const inputName = GAMEPAD_BUTTONS[buttonIndex];

      if (isCurrentlyPressed) {
        if (!wasPressedLastFrame) {
          // If button is pressed but wasn't pressed last frame, fire the down event
          fireEvent({
            blockingListeners,
            eventName: getEventName({
              eventType: GamepadButtonEvent.onDown,
              inputName,
            }),
            gamepadIndex,
            listeners,
          });
        }

        // If button is pressed, fire on held event
        fireEvent({
          blockingListeners,
          eventName: getEventName({
            eventType: GamepadButtonEvent.onHeld,
            inputName,
          }),
          gamepadIndex,
          listeners,
        });
      } else if (wasPressedLastFrame) {
        // If button isn't pressed but was last frame, fire the up event
        fireEvent({
          blockingListeners,
          eventName: getEventName({
            eventType: GamepadButtonEvent.onUp,
            inputName,
          }),
          gamepadIndex,
          listeners,
        });
      }

      // Update new state to reflect gamepad state
      const newGamepadState = newState.get(gamepadIndex);
      if (newGamepadState && isCurrentlyPressed !== wasPressedLastFrame) {
        wasStateUpdated = true;
        const pressedButtons = [...newGamepadState.pressedButtons];
        pressedButtons[buttonIndex] = isCurrentlyPressed;
        newState.set(gamepadIndex, { pressedButtons });
      }
    });
  }

  return wasStateUpdated ? newState : oldState;
}

export function updateGamepadState(
  state: Map<number, GamepadState>,
  action: GamepadReducerActions,
): Map<number, GamepadState> {
  // Note that we rely on the browser to properly maintain gamepad state
  // and indices when gamepads connect and disconnect so we're just passing
  // through values from the event itself and keeping our internal state
  // consistent with the browser's.
  switch (action.type) {
    case 'addGamepad':
      const gamepadState = new Map(state);
      gamepadState.set(action.gamepadIndex, {
        pressedButtons: Array<boolean>(GAMEPAD_BUTTONS.length).fill(false),
      });
      return gamepadState;
    case 'removeGamepad':
      const newState = new Map(state);
      newState.delete(action.gamepadIndex);
      return newState;
    case 'updateGamepads':
      return updateGamepad({
        blockingListeners: action.blockingListeners,
        gamepads: action.gamepads,
        listeners: action.listeners,
        oldState: state,
      });
  }
}

export const GamepadInputBase: FC = ({ children }) => {
  const blockingListeners = useRef<BlockingListeners>({});
  const listeners = useRef<Listeners>({});
  const animationRequestRef = useRef<number>(0);
  const gamepadState = useRef<Map<number, GamepadState>>(new Map());

  const subscribeToInput = useCallback(
    ({ callback, eventName, preventDefault }: SubscribeToInputOpts) => {
      if (!isBrowser()) {
        return () => undefined;
      }

      if (preventDefault) {
        if (blockingListeners.current[eventName]) {
          logger.warn({
            category: 'GamepadInputRoot',
            message: `Attempted registry of multiple blocking listeners on ${eventName}`,
            package: 'gamepad-input',
          });

          return () => undefined;
        }
        blockingListeners.current[eventName] = callback;
        return () => {
          blockingListeners.current[eventName] = undefined;
        };
      }
      listeners.current[eventName] ??= new Set();
      listeners.current[eventName]?.add(callback);
      return () => {
        listeners.current[eventName]?.delete(callback);
      };
    },
    [],
  );

  const value = useMemo(
    () => ({
      subscribeToGamepadAxisInputEvent: ({
        axis,
        callback,
        preventDefault,
      }: SubscribeToGamepadAxisInputEventOpts) => {
        const eventName = getEventName({
          eventType: GamepadAxisEvent.onValueChanged,
          inputName: axis,
        });

        return subscribeToInput({ callback, eventName, preventDefault });
      },
      subscribeToGamepadAxisInputEvents: (
        inputListeners: SubscribeToGamepadAxisInputEventOpts[],
      ) => {
        const listenersToRemove: Array<() => void> = [];
        inputListeners.forEach(({ axis, callback, preventDefault }) => {
          const eventName = getEventName({
            eventType: GamepadAxisEvent.onValueChanged,
            inputName: axis,
          });
          listenersToRemove.push(
            subscribeToInput({ callback, eventName, preventDefault }),
          );
        });
        return () => {
          listenersToRemove.forEach((removeListener) => {
            removeListener();
          });
        };
      },
      subscribeToGamepadButtonInputEvent: ({
        button,
        callback,
        eventType,
        preventDefault,
      }: SubscribeToGamepadButtonInputEventOpts) => {
        const eventName = getEventName({
          eventType,
          inputName: button,
        });

        return subscribeToInput({ callback, eventName, preventDefault });
      },
      subscribeToGamepadButtonInputEvents: (
        inputListeners: SubscribeToGamepadButtonInputEventOpts[],
      ) => {
        const listenersToRemove: Array<() => void> = [];
        inputListeners.forEach(
          ({ button, callback, eventType, preventDefault }) => {
            const eventName = getEventName({ eventType, inputName: button });
            listenersToRemove.push(
              subscribeToInput({ callback, eventName, preventDefault }),
            );
          },
        );
        return () => {
          listenersToRemove.forEach((removeListener) => {
            removeListener();
          });
        };
      },
    }),
    [subscribeToInput],
  );

  // Callback that will handle our gamepad "update loop" which is scheduled
  // each frame using `requestAnimationFrame`
  const updateGamepads = useCallback(() => {
    gamepadState.current = updateGamepadState(gamepadState.current, {
      blockingListeners: blockingListeners.current,
      gamepads: window.navigator.getGamepads(),
      listeners: listeners.current,
      type: 'updateGamepads',
    });
    animationRequestRef.current = window.requestAnimationFrame(updateGamepads);
  }, []);

  useEffect(() => {
    // istanbul ignore next: trivial
    const onGamepadConnect = (event: GamepadEvent) => {
      // Initialize with new unpressed state when gamepad connects
      gamepadState.current = updateGamepadState(gamepadState.current, {
        gamepadIndex: event.gamepad.index,
        type: 'addGamepad',
      });
      animationRequestRef.current =
        window.requestAnimationFrame(updateGamepads);
    };

    // istanbul ignore next: trivial
    const onGamepadDisconnect = (event: GamepadEvent) => {
      // Clear out previous button pressed state if gamepad disconnects
      gamepadState.current = updateGamepadState(gamepadState.current, {
        gamepadIndex: event.gamepad.index,
        type: 'removeGamepad',
      });
    };

    window.addEventListener('gamepadconnected', onGamepadConnect);
    window.addEventListener('gamepaddisconnected', onGamepadDisconnect);
    return () => {
      window.removeEventListener('gamepadconnected', onGamepadConnect);
      window.removeEventListener('gamepaddisconnected', onGamepadDisconnect);
      window.cancelAnimationFrame(animationRequestRef.current);
    };
  }, [updateGamepads]);

  return <gamepadInputContext.Provider children={children} value={value} />;
};
GamepadInputBase.displayName = 'GamepadInputBase';

export const GamepadInputRoot: FC = ({ children }) => {
  const {
    common: { platform },
  } = useStaticEnvironment();
  if (!GamepadUnsupportedPlatform.includes(platform)) {
    return <GamepadInputBase children={children} />;
  }
  return <>{children}</>;
};
GamepadInputRoot.displayName = 'GamepadInputRoot';

// istanbul ignore next: trivial
export const getEventName = ({
  eventType,
  inputName,
}: GetEventNameOpts): GamepadEventName => {
  return `gamepad-input-${inputName}-${eventType}`;
};
