import { act } from 'react-dom/test-utils';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { GAMEPAD_AXES, GAMEPAD_BUTTONS } from '../constants';
import { getMockGamepad } from '../test-mocks';
import type { GamepadState } from '../types';
import { GamepadAxisEvent, GamepadButtonEvent } from '../types';
import { fireEvent } from './fireEvent';
import {
  GamepadInputRoot,
  getEventName,
  updateGamepad,
  updateGamepadState,
} from '.';

jest.mock('./fireEvent', () => ({
  fireEvent: jest.fn(),
}));

const mockFireEvent = fireEvent as jest.Mock;

describe(GamepadInputRoot, () => {
  describe('windowEventListeners', () => {
    let spyAddEvent: any;
    let spyRemoveEvent: any;
    let spyCancelAnimFrame: any;

    const setup = createMountWrapperFactory(GamepadInputRoot);

    it('adds event listeners when component mounts', () => {
      spyAddEvent = jest.spyOn(window, 'addEventListener');
      spyRemoveEvent = jest.spyOn(window, 'removeEventListener');
      spyCancelAnimFrame = jest.spyOn(window, 'cancelAnimationFrame');
      setup();

      // Verify all the event listeners were subscribed to
      expect(spyAddEvent).toHaveBeenCalledWith(
        'gamepadconnected',
        expect.anything(),
      );
      expect(spyAddEvent).toHaveBeenCalledWith(
        'gamepaddisconnected',
        expect.anything(),
      );

      // Since we haven't unmounted yet, verify none of the unsubscription methods have been called
      expect(spyRemoveEvent).not.toHaveBeenCalledWith(
        'gamepadconnected',
        expect.anything(),
      );
      expect(spyRemoveEvent).not.toHaveBeenCalledWith(
        'gamepaddisconnected',
        expect.anything(),
      );
    });

    it('removes event listeners when component unmounts', () => {
      const { wrapper } = setup();

      // Verify none of the listeners are unsubscribed until we umount
      expect(spyRemoveEvent).not.toHaveBeenCalledWith(
        'gamepadconnected',
        expect.anything(),
      );
      expect(spyRemoveEvent).not.toHaveBeenCalledWith(
        'gamepaddisconnected',
        expect.anything(),
      );
      expect(spyCancelAnimFrame).not.toHaveBeenCalled();

      act(() => {
        wrapper.unmount();
      });

      // After unmounting, verify we unsubscribed
      expect(spyRemoveEvent).toHaveBeenCalledWith(
        'gamepadconnected',
        expect.anything(),
      );
      expect(spyRemoveEvent).toHaveBeenCalledWith(
        'gamepaddisconnected',
        expect.anything(),
      );
      expect(spyCancelAnimFrame).toHaveBeenCalled();
    });
  });

  describe('updateGamepad', () => {
    it('emits events when button input is detected', () => {
      const blockingListeners = {};
      const listeners = {};
      const PRESSED_BUTTON_INDEX = 2;
      const onHeldEventName = getEventName({
        eventType: GamepadButtonEvent.onHeld,
        inputName: GAMEPAD_BUTTONS[PRESSED_BUTTON_INDEX],
      });
      const onDownEventName = getEventName({
        eventType: GamepadButtonEvent.onDown,
        inputName: GAMEPAD_BUTTONS[PRESSED_BUTTON_INDEX],
      });
      const onUpEventName = getEventName({
        eventType: GamepadButtonEvent.onUp,
        inputName: GAMEPAD_BUTTONS[PRESSED_BUTTON_INDEX],
      });
      const initialState = new Map();
      initialState.set(0, {
        pressedButtons: new Array<boolean>(GAMEPAD_BUTTONS.length).fill(false),
      });
      const noInputState = updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [])],
        listeners,
        oldState: initialState,
      });
      expect(mockFireEvent).not.toHaveBeenCalled();

      // simulate button press
      const initialButtonPressState = updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [PRESSED_BUTTON_INDEX])],
        listeners,
        oldState: noInputState,
      });
      expect(mockFireEvent).toHaveBeenNthCalledWith(1, {
        blockingListeners,
        eventName: onDownEventName,
        gamepadIndex: 0,
        listeners,
      });
      expect(mockFireEvent).toHaveBeenNthCalledWith(2, {
        blockingListeners,
        eventName: onHeldEventName,
        gamepadIndex: 0,
        listeners,
      });

      // button is still pressed next frame
      const buttonHeldState = updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [PRESSED_BUTTON_INDEX])],
        listeners,
        oldState: initialButtonPressState,
      });
      expect(mockFireEvent).toHaveBeenNthCalledWith(3, {
        blockingListeners,
        eventName: onHeldEventName,
        gamepadIndex: 0,
        listeners,
      });

      // simulate button release
      updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [])],
        listeners,
        oldState: buttonHeldState,
      });
      expect(mockFireEvent).toHaveBeenNthCalledWith(4, {
        blockingListeners,
        eventName: onUpEventName,
        gamepadIndex: 0,
        listeners,
      });
    });

    it('emits events when axis input is detected', () => {
      const blockingListeners = {};
      const listeners = {};
      const AXIS_TO_CHANGE = 2;
      const AXIS_CHANGED_VALUE = 0.9;
      const eventName = getEventName({
        eventType: GamepadAxisEvent.onValueChanged,
        inputName: GAMEPAD_AXES[AXIS_TO_CHANGE],
      });
      // Axis input events do not rely on previous values so oldState
      // is not relevant for these tests and we can safely reuse the
      // value of our initial state.
      const mockState = new Map();
      mockState.set(0, {
        pressedButtons: new Array<boolean>(GAMEPAD_BUTTONS.length).fill(false),
      });

      updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [], [0, 0, 0, 0])],
        listeners,
        oldState: mockState,
      });
      expect(mockFireEvent).not.toHaveBeenCalled();

      // simulate joystick movement
      updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [], [0, 0, AXIS_CHANGED_VALUE, 0])],
        listeners,
        oldState: mockState,
      });
      expect(mockFireEvent).toHaveBeenNthCalledWith(1, {
        blockingListeners,
        eventName,
        gamepadIndex: 0,
        listeners,
        value: AXIS_CHANGED_VALUE,
      });

      // joystick maintains value
      updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [], [0, 0, AXIS_CHANGED_VALUE, 0])],
        listeners,
        oldState: mockState,
      });
      expect(mockFireEvent).toHaveBeenNthCalledWith(2, {
        blockingListeners,
        eventName,
        gamepadIndex: 0,
        listeners,
        value: AXIS_CHANGED_VALUE,
      });

      // joystick is released
      updateGamepad({
        blockingListeners,
        gamepads: [getMockGamepad(0, [])],
        listeners,
        oldState: mockState,
      });
      expect(mockFireEvent).toHaveBeenCalledTimes(2);
    });
  });

  describe('gamepadStateReducer', () => {
    it("can add new gamepads and can't add the same gamepad twice", () => {
      // Add a new gamepad, verify a new empty array initialized all to false
      const singleGamepadState = updateGamepadState(new Map(), {
        gamepadIndex: 2,
        type: 'addGamepad',
      });
      expect(singleGamepadState.size).toEqual(1);
      expect(singleGamepadState.get(2)?.pressedButtons.length).toEqual(
        GAMEPAD_BUTTONS.length,
      );
      singleGamepadState.get(2)?.pressedButtons.forEach((entry) => {
        expect(entry).toEqual(false);
      });

      // Add a new gamepad, verify a new empty array initialized all to false
      const twoGamepadState = updateGamepadState(singleGamepadState, {
        gamepadIndex: 1,
        type: 'addGamepad',
      });
      expect(twoGamepadState.size).toEqual(2);
      expect(twoGamepadState.get(1)?.pressedButtons.length).toEqual(
        GAMEPAD_BUTTONS.length,
      );
      twoGamepadState.get(1)?.pressedButtons.forEach((entry) => {
        expect(entry).toEqual(false);
      });
    });

    it('removes state if state exists', () => {
      const threeGamepadState = new Map<number, GamepadState>();
      threeGamepadState.set(1, {
        pressedButtons: new Array<boolean>(GAMEPAD_BUTTONS.length).fill(false),
      });
      threeGamepadState.set(2, {
        pressedButtons: new Array<boolean>(GAMEPAD_BUTTONS.length).fill(false),
      });
      threeGamepadState.set(3, {
        pressedButtons: new Array<boolean>(GAMEPAD_BUTTONS.length).fill(false),
      });

      // Remove a gamepad by index, verify that the new state has it removed
      const twoGamepadState = updateGamepadState(threeGamepadState, {
        gamepadIndex: 2,
        type: 'removeGamepad',
      });
      expect(twoGamepadState.size).toEqual(2);
      expect(twoGamepadState.has(1)).toEqual(true);
      expect(twoGamepadState.has(2)).toEqual(false);
      expect(twoGamepadState.has(3)).toEqual(true);

      // Verify that removing a value that didn't exist is a noop
      const attemptInvalidRemovalState = updateGamepadState(twoGamepadState, {
        gamepadIndex: 2,
        type: 'removeGamepad',
      });
      expect(attemptInvalidRemovalState.size).toEqual(2);
      expect(attemptInvalidRemovalState.has(1)).toEqual(true);
      expect(attemptInvalidRemovalState.has(2)).toEqual(false);
      expect(attemptInvalidRemovalState.has(3)).toEqual(true);
    });
  });
});
