export enum GamepadButton {
  DpadDown = 'dpad-down',
  DpadLeft = 'dpad-left',
  DpadRight = 'dpad-right',
  DpadUp = 'dpad-up',
  FaceButtonDown = 'face-button-down',
  FaceButtonLeft = 'face-button-left',
  FaceButtonRight = 'face-button-right',
  FaceButtonUp = 'face-button-up',
  L1 = 'l1',
  L2 = 'l2',
  L3 = 'l3',
  R1 = 'r1',
  R2 = 'r2',
  R3 = 'r3',
  Select = 'select',
  Start = 'start',
}

export enum GamepadAxis {
  LeftStickHorizontal = 'left-stick-horizontal',
  LeftStickVertical = 'left-stick-vertical',
  RightStickHorizontal = 'right-stick-horizontal',
  RightStickVertical = 'right-stick-vertical',
}

export enum GamepadButtonEvent {
  /**
   * Fired one time on the frame the button is first pressed.
   * Will be fired before the `onHeld` event.
   */
  onDown = 'onDown',
  /**
   * Fired on every frame while the button is pressed. Will be
   * fired after the `onDown` event on the first frame where
   * the button is pressed.
   */
  onHeld = 'onHeld',
  /**
   * Fired one time on the frame the button is first released.
   */
  onUp = 'onUp',
}

export enum GamepadAxisEvent {
  onValueChanged = 'onValueChanged',
}

export type GamepadActionFunc = (gamepadIndex: number, value?: number) => void;

export type BlockingListeners = {
  [key in GamepadEventName]?: GamepadActionFunc;
};

export type Listeners = {
  [key in GamepadEventName]?: Set<GamepadActionFunc>;
};

// Arguments used by our reducer that takes in the old gamepad state plus
// the current gamepad state and fires the appropriate callbacks plus provides
// a new state. When gamepads are disconnected, they're replaced with a null
// placeholder to maintain indices for other gamepads.
export type UpdateGamepadReducerOpts = {
  blockingListeners: BlockingListeners;
  gamepads: (Gamepad | null)[];
  listeners: Listeners;
  oldState: Map<number, GamepadState>;
};

// We maintain a snapshot of whether or not each button is pressed for each
// gamepad, and each time we update the gamepad, we use this previous state
// plus the current state to determine if we should fire down/up events,
// since we only want to fire that event if the button state changed in the
// most recent frame
export type GamepadState = {
  pressedButtons: boolean[];
};

export type GamepadReducerActions =
  | {
      blockingListeners: BlockingListeners;
      gamepads: (Gamepad | null)[];
      listeners: Listeners;
      type: 'updateGamepads';
    }
  | { gamepadIndex: number; type: 'addGamepad' }
  | { gamepadIndex: number; type: 'removeGamepad' };

export type SubscribeToInputOpts = {
  callback: GamepadActionFunc;
  eventName: GamepadEventName;
  preventDefault?: boolean | undefined;
};

export type GetEventNameOpts =
  | {
      eventType: GamepadAxisEvent;
      inputName: GamepadAxis;
    }
  | {
      eventType: GamepadButtonEvent;
      inputName: GamepadButton;
    };

// Note that each event type should only be used with the corresponding
// input type (Button/Axis). The function we use to generate the event
// names is only used internally for this package and we restrict the
// params to be consistent in `GetEventNameOpts` so it shouldn't cause
// any issues. At worst we end up with listeners registered that will
// never be invoked.
export type GamepadEventName =
  | `gamepad-input-${GamepadAxis | GamepadButton}-${
      | GamepadAxisEvent
      | GamepadButtonEvent}`;
