import { createContext, useContext } from 'react';
import type {
  GamepadActionFunc,
  GamepadAxis,
  GamepadButton,
  GamepadButtonEvent,
} from '../types';

type GamepadInputContext = {
  subscribeToGamepadAxisInputEvent: SubscribeToGamepadAxisInputEvent;
  subscribeToGamepadAxisInputEvents: SubscribeToGamepadAxisInputEvents;
  subscribeToGamepadButtonInputEvent: SubscribeToGamepadButtonInputEvent;
  subscribeToGamepadButtonInputEvents: SubscribeToGamepadButtonInputEvents;
};

export type SubscribeToGamepadAxisInputEventOpts = {
  /**
   * The axis triggering the callback
   */
  axis: GamepadAxis;
  /**
   * Callback to be executed when the input is detected
   */
  callback: GamepadActionFunc;
  /**
   * If this is marked true, other listeners will not be fired for this input type.
   * Only one blocking listener is allowed for each input type.
   */
  preventDefault?: true;
};
export type SubscribeToGamepadButtonInputEventOpts = {
  /**
   * The button triggering the callback
   */
  button: GamepadButton;
  /**
   * Callback to be executed when the input is detected
   */
  callback: GamepadActionFunc;
  /**
   * The event type to subscribe to.
   */
  eventType: GamepadButtonEvent;
  /**
   * If this is marked true, other listeners will not be fired for this input type.
   * Only one blocking listener is allowed for each input type.
   */
  preventDefault?: true;
};

/**
 * @returns a callback that will remove the listener. This works well with useEffect
 */
type SubscribeToGamepadButtonInputEvent = (
  subscribeToGamepadButtonInputOpts: SubscribeToGamepadButtonInputEventOpts,
) => () => void;

type SubscribeToGamepadAxisInputEvent = (
  subscribeToGamepadButtonInputOpts: SubscribeToGamepadAxisInputEventOpts,
) => () => void;

type SubscribeToGamepadButtonInputEvents = (
  inputListeners: SubscribeToGamepadButtonInputEventOpts[],
) => () => void;

type SubscribeToGamepadAxisInputEvents = (
  inputListeners: SubscribeToGamepadAxisInputEventOpts[],
) => () => void;

// istanbul ignore next: trivial
export const gamepadInputContext = createContext<GamepadInputContext>({
  subscribeToGamepadAxisInputEvent: () => () => undefined,
  subscribeToGamepadAxisInputEvents: () => () => undefined,
  subscribeToGamepadButtonInputEvent: () => () => undefined,
  subscribeToGamepadButtonInputEvents: () => () => undefined,
});

// istanbul ignore next: trivial
/**
 * Hook used for adding listeners to gamepad inputs
 *
 * @returns
 * - {@link SubscribeToGamepadInput subscribeToGamepadInput} - used for subscribing to a gamepad input event
 */
export function useGamepadInput(): GamepadInputContext {
  return useContext(gamepadInputContext);
}
