import { renderHook } from '@testing-library/react-hooks';
import { GamepadInputRoot } from '../GamepadInputRoot';
import { getMockGamepad } from '../test-mocks';
import { GamepadAxis, GamepadButton, GamepadButtonEvent } from '../types';
import { useGamepadInput } from '.';

jest.mock('tachyon-logger', () => ({ logger: { warn: jest.fn() } }));
declare global {
  interface GamepadEvent extends Event {
    readonly gamepad: Gamepad;
  }
}

const mockGamepadsOnWindow = (mockReturnValue: (Gamepad | null)[]) => {
  window.navigator.getGamepads = () => mockReturnValue;
};

describe(useGamepadInput, () => {
  beforeEach(() => {
    window.GamepadEvent = jest.fn((type, eventInit) => {
      const event = new Event(type) as GamepadEvent;
      (event as any).gamepad = eventInit;
      return event;
    });
  });

  afterEach(() => {
    delete (window as any).GamepadEvent;
  });
  describe('subscribeToIntent', () => {
    it('adds and removes a single button listener', () => {
      jest.useFakeTimers();
      const { result } = renderHook(useGamepadInput, {
        wrapper: GamepadInputRoot,
      });

      window.dispatchEvent(
        new GamepadEvent('gamepadconnected', {
          gamepad: getMockGamepad(0, []),
        }),
      );

      const { subscribeToGamepadButtonInputEvent } = result.current;
      const mockCallback = jest.fn();
      const persistedMockCallback = jest.fn();
      const removeListener = subscribeToGamepadButtonInputEvent({
        button: GamepadButton.FaceButtonLeft,
        callback: mockCallback,
        eventType: GamepadButtonEvent.onDown,
      });
      subscribeToGamepadButtonInputEvent({
        button: GamepadButton.FaceButtonLeft,
        callback: persistedMockCallback,
        eventType: GamepadButtonEvent.onDown,
      });
      mockGamepadsOnWindow([getMockGamepad(0, [])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).not.toHaveBeenCalled();
      expect(mockCallback).not.toHaveBeenCalled();

      mockGamepadsOnWindow([getMockGamepad(0, [2])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).toHaveBeenCalledTimes(1);
      expect(mockCallback).toHaveBeenCalledTimes(1);

      removeListener();
      mockGamepadsOnWindow([getMockGamepad(0, [2])]);
      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).toHaveBeenCalledTimes(2);
      expect(mockCallback).toHaveBeenCalledTimes(1);
      jest.clearAllTimers();
    });

    it('adds and removes multiple button listeners', () => {
      jest.useFakeTimers();
      const { result } = renderHook(useGamepadInput, {
        wrapper: GamepadInputRoot,
      });

      window.dispatchEvent(
        new GamepadEvent('gamepadconnected', {
          gamepad: getMockGamepad(0, []),
        }),
      );

      const { subscribeToGamepadButtonInputEvents } = result.current;
      const mockCallback = jest.fn();
      const persistedMockCallback = jest.fn();
      const removeListener = subscribeToGamepadButtonInputEvents([
        {
          button: GamepadButton.FaceButtonLeft,
          callback: mockCallback,
          eventType: GamepadButtonEvent.onDown,
        },
        {
          button: GamepadButton.FaceButtonLeft,
          callback: persistedMockCallback,
          eventType: GamepadButtonEvent.onDown,
        },
      ]);
      mockGamepadsOnWindow([getMockGamepad(0, [])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).not.toHaveBeenCalled();
      expect(mockCallback).not.toHaveBeenCalled();

      mockGamepadsOnWindow([getMockGamepad(0, [2])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).toHaveBeenCalledTimes(1);
      expect(mockCallback).toHaveBeenCalledTimes(1);

      removeListener();
      mockGamepadsOnWindow([getMockGamepad(0, [2])]);
      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).toHaveBeenCalledTimes(1);
      expect(mockCallback).toHaveBeenCalledTimes(1);
      jest.clearAllTimers();
    });

    it('adds and removes an axis listener', () => {
      jest.useFakeTimers();
      const { result } = renderHook(useGamepadInput, {
        wrapper: GamepadInputRoot,
      });

      window.dispatchEvent(
        new GamepadEvent('gamepadconnected', {
          gamepad: getMockGamepad(0, []),
        }),
      );

      const { subscribeToGamepadAxisInputEvent } = result.current;
      const mockCallback = jest.fn();
      const persistedMockCallback = jest.fn();
      const removeListener = subscribeToGamepadAxisInputEvent({
        axis: GamepadAxis.LeftStickHorizontal,
        callback: mockCallback,
      });
      subscribeToGamepadAxisInputEvent({
        axis: GamepadAxis.LeftStickHorizontal,
        callback: persistedMockCallback,
      });
      mockGamepadsOnWindow([getMockGamepad(0, [])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).not.toHaveBeenCalled();
      expect(mockCallback).not.toHaveBeenCalled();

      mockGamepadsOnWindow([getMockGamepad(0, [], [90])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).toHaveBeenCalledTimes(1);
      expect(mockCallback).toHaveBeenCalledTimes(1);

      removeListener();
      mockGamepadsOnWindow([getMockGamepad(0, [], [90])]);
      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).toHaveBeenCalledTimes(2);
      expect(mockCallback).toHaveBeenCalledTimes(1);
      jest.clearAllTimers();
    });

    it('adds and removes multiple axis listeners', () => {
      jest.useFakeTimers();
      const { result } = renderHook(useGamepadInput, {
        wrapper: GamepadInputRoot,
      });

      window.dispatchEvent(
        new GamepadEvent('gamepadconnected', {
          gamepad: getMockGamepad(0, []),
        }),
      );

      const { subscribeToGamepadAxisInputEvents } = result.current;
      const firstMock = jest.fn();
      const secondMock = jest.fn();
      const removeListener = subscribeToGamepadAxisInputEvents([
        {
          axis: GamepadAxis.LeftStickHorizontal,
          callback: firstMock,
        },
        {
          axis: GamepadAxis.LeftStickHorizontal,
          callback: secondMock,
        },
      ]);
      mockGamepadsOnWindow([getMockGamepad(0, [])]);

      jest.runOnlyPendingTimers();
      expect(secondMock).not.toHaveBeenCalled();
      expect(firstMock).not.toHaveBeenCalled();

      mockGamepadsOnWindow([getMockGamepad(0, [], [90])]);

      jest.runOnlyPendingTimers();
      expect(secondMock).toHaveBeenCalledTimes(1);
      expect(firstMock).toHaveBeenCalledTimes(1);

      removeListener();
      mockGamepadsOnWindow([getMockGamepad(0, [], [90])]);
      jest.runOnlyPendingTimers();
      expect(secondMock).toHaveBeenCalledTimes(1);
      expect(firstMock).toHaveBeenCalledTimes(1);
      jest.clearAllTimers();
    });

    it('prefers blocking listeners if they exist', () => {
      jest.useFakeTimers();
      const { result } = renderHook(useGamepadInput, {
        wrapper: GamepadInputRoot,
      });

      window.dispatchEvent(
        new GamepadEvent('gamepadconnected', {
          gamepad: getMockGamepad(0, []),
        }),
      );

      const { subscribeToGamepadButtonInputEvent } = result.current;
      const blockingCallback = jest.fn();
      const persistedMockCallback = jest.fn();
      const removeListener = subscribeToGamepadButtonInputEvent({
        button: GamepadButton.FaceButtonLeft,
        callback: blockingCallback,
        eventType: GamepadButtonEvent.onDown,
        preventDefault: true,
      });
      subscribeToGamepadButtonInputEvent({
        button: GamepadButton.FaceButtonLeft,
        callback: persistedMockCallback,
        eventType: GamepadButtonEvent.onDown,
      });
      mockGamepadsOnWindow([getMockGamepad(0, [])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).not.toHaveBeenCalled();
      expect(blockingCallback).not.toHaveBeenCalled();

      mockGamepadsOnWindow([getMockGamepad(0, [2])]);

      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).not.toHaveBeenCalled();
      expect(blockingCallback).toHaveBeenCalledTimes(1);

      removeListener();
      mockGamepadsOnWindow([getMockGamepad(0, [2])]);
      jest.runOnlyPendingTimers();
      expect(persistedMockCallback).toHaveBeenCalled();
      expect(blockingCallback).toHaveBeenCalledTimes(1);
      jest.clearAllTimers();
    });

    it('prevents adding multiple blocking listeners to the same event', () => {
      jest.useFakeTimers();
      const { result } = renderHook(useGamepadInput, {
        wrapper: GamepadInputRoot,
      });

      window.dispatchEvent(
        new GamepadEvent('gamepadconnected', {
          gamepad: getMockGamepad(0, []),
        }),
      );

      const { subscribeToGamepadButtonInputEvent } = result.current;
      const blockingCallback = jest.fn();
      const invalidBlockingCallback = jest.fn();
      subscribeToGamepadButtonInputEvent({
        button: GamepadButton.FaceButtonLeft,
        callback: blockingCallback,
        eventType: GamepadButtonEvent.onDown,
        preventDefault: true,
      });
      subscribeToGamepadButtonInputEvent({
        button: GamepadButton.FaceButtonLeft,
        callback: invalidBlockingCallback,
        eventType: GamepadButtonEvent.onDown,
        preventDefault: true,
      });
      mockGamepadsOnWindow([getMockGamepad(0, [])]);

      jest.runOnlyPendingTimers();
      expect(invalidBlockingCallback).not.toHaveBeenCalled();
      expect(blockingCallback).not.toHaveBeenCalled();

      mockGamepadsOnWindow([getMockGamepad(0, [2])]);

      jest.runOnlyPendingTimers();
      expect(invalidBlockingCallback).not.toHaveBeenCalled();
      expect(blockingCallback).toHaveBeenCalledTimes(1);
      jest.clearAllTimers();
    });
  });
});
