import {
  ROOT_FOCUS_ID,
  childFocusIndexForFocusedArea,
  focusIdContainsElement,
  getFocusId,
  splitFocusId,
} from './utils';

export { getFocusId, ROOT_FOCUS_ID } from './utils';

type BeforeNavHandler = {
  /**
   * When used on a navigation area this will precede the tv-nav input handler.
   * The callback can return `true` to preclude the default behavior.
   */
  onDown: (() => true | void) | undefined;
  /**
   * When used on a navigation area this will precede the tv-nav input handler.
   * The callback can return `true` to preclude the default behavior.
   */
  onLeft: (() => true | void) | undefined;
  /**
   * When used on a navigation area this will precede the tv-nav input handler.
   * The callback can return `true` to preclude the default behavior.
   */
  onRight: (() => true | void) | undefined;
  /**
   * When used on a navigation area this will precede the tv-nav input handler.
   * The callback can return `true` to preclude the default behavior.
   */
  onUp: (() => true | void) | undefined;
};

export type FocusableAreaElement = Partial<BeforeNavHandler> & {
  /**
   * The index of currently focused item in this navigation area, retained
   * through focus and blur events.
   */
  childFocusIndex: number;
  /**
   * Mandatory for focus areas to make debugging easier.
   */
  debugLabel: string;
  /**
   * Total number of navigable child elements that will be rendered.
   */
  elementCount: number;
  /**
   * Unique ID used to represent the element by its position within a focus tree.
   */
  focusId: string;
  /**
   * The focus index assigned to the navigation element.
   */
  focusIndex: number;
  /**
   * Whether or not wheel-based navigation is enabled for the area.
   */
  handleWheel: boolean;
  /**
   * The amount by which to increase the focusIndex in response to horizontal
   * navigation events.
   */
  horizontalIncrement: number;
  /**
   * Unique ID generated for the specific instance of the area being registered.
   * Should be regenerated any time the area changes / is re-registered.
   */
  instanceId: string;
  /**
   * The number of visible elements in a "page" of this navigation area. Setting this
   * to anything above 0 turns on page-navigation-event handling for this area.
   */
  pageSize?: number | undefined;
  parentFocusId: string;
  /**
   * Allows a navigation area to take focus upon its first render into the page.
   * This can be used for things like ensuring that the primary area of a page
   * is focused after a loading transition or for modals to take focus once
   * they come into the page.
   */
  takeFocusOnFirstRender?: boolean | undefined;
  type: 'grid' | 'horizontal' | 'node' | 'vertical';
  /**
   * The amount by which to increase the focusIndex in response to vertical
   * navigation events.
   */
  verticalIncrement: number;
  /**
   * Whether or not the area should be persisted until its parent is removed.
   * Useful if the area can be temporarily removed such as with virtualization.
   */
  virtualizable?: boolean | undefined;
};

export interface FocusBroadcasterMethods {
  /**
   * Adds a change listener that will be invoked any time focus changes to a
   * new element.
   */
  addFocusChangeListener: (listener: () => void) => () => void;
  /**
   * Use to imperatively focus an element.
   */
  focusElement: (elementFocusId: string) => void;
  /**
   * Returns information about a focus area element for a given area focus ID.
   */
  getAreaElement: (areaFocusId: string) => FocusableAreaElement;
  /**
   * Use to determine whether or not an element is focused.
   */
  isFocused: (parentFocusId: string, focusIndex: number) => boolean;
  /**
   * Registers a navigation area with the focus system.
   */
  registerArea: (area: FocusableAreaElement) => void;
  /**
   * Removes a navigation area from the focus system.
   */
  removeArea: (area: FocusableAreaElement) => void;
}

type FocusId = string;

export class FocusBroadcaster implements FocusBroadcasterMethods {
  /**
   * A unique ID indicating the element that is currently focused. Should
   * always be a leaf in the navigation tree.
   */
  private currentFocusId: FocusId = ROOT_FOCUS_ID;

  /**
   * A mapping of focus ID to registered area elements.
   *
   * A flat map was chosen rather than a tree of nested objects to:
   * - avoid overall complexity of tree management
   * - avoid having to deep clone / merge to replace or remove areas
   * - fast lookups by ID which happen frequently in the system
   */
  private focusAreas: Record<FocusId, FocusableAreaElement> = {};

  /**
   * Listeners that should be notified when the currentFocusId value changes.
   * These are keyed off of listenerIncrement and aren't specifically tied
   * to any focus ID.
   */
  private focusChangeListeners: Record<string, () => void> = {};

  /**
   * Stores last known child focus indices for areas that have been removed
   * but may potentially be re-created due to virtualization.
   */
  private virtualizedAreaChildFocusIndices: Record<string, number> = {};

  /**
   * Serves as unique ID for registered listeners. Incremented each time a new
   * listener is added. Used for clean-up during unsubscription.
   */
  private listenerIncrement = 0;

  public isFocused = (parentFocusId: string, focusIndex: number): boolean => {
    return focusIdContainsElement({
      elementId: getFocusId(parentFocusId, focusIndex),
      focusId: this.currentFocusId,
    });
  };

  public getAreaElement = (focusId: string): FocusableAreaElement => {
    const area = this.focusAreas[focusId];
    if (!area) {
      throw new Error(`No area found for ${focusId}`);
    }

    return area;
  };

  public getFocusedAreaElement = (): FocusableAreaElement => {
    const { parentFocusId: focusedAreaId } = splitFocusId(this.currentFocusId);
    return this.getAreaElement(focusedAreaId);
  };

  public focusElement = (newFocusId: string): void => {
    const newDeepestFocusId = this.findDeepestFocusId(newFocusId);
    if (this.currentFocusId === newDeepestFocusId) {
      return;
    }

    this.currentFocusId = newDeepestFocusId;

    // Reconcile all navigation areas that are part of the new focus ID ensuring
    // that they have the correct childFocusIndex values. This is to account for
    // focus changes that happen outside of normal arrow key navigation.
    Object.values(this.focusAreas).forEach((navArea) => {
      if (
        focusIdContainsElement({
          elementId: navArea.focusId,
          focusId: this.currentFocusId,
        })
      ) {
        navArea.childFocusIndex = childFocusIndexForFocusedArea(
          this.currentFocusId,
          navArea.focusId,
        );
      }
    });

    this.broadcastFocusChange();
  };

  public registerArea = (areaToRegister: FocusableAreaElement): void => {
    const { focusIndex, parentFocusId } = areaToRegister;
    if (parentFocusId === ROOT_FOCUS_ID && focusIndex !== 0) {
      throw new Error('Only a single root element may be registered');
    }

    const areaToReplace = this.focusAreas[areaToRegister.focusId];

    let initialChildFocusIndex = areaToRegister.childFocusIndex;
    if (areaToReplace) {
      // Preserve existing child focus since the area is presumably the same (naive type comparison)
      // and it is updating its element count.
      // TODO: Only override the registered area's childFocusIndex if existing / persisted area have
      // a matching focusId+key combo to eliminate possibility of starting
      // on an unexpected child focus index if existing area is replaced with a new
      // area of the same type but with completely different content.
      if (areaToReplace.type === areaToRegister.type) {
        initialChildFocusIndex = areaToReplace.childFocusIndex;
      }
    } else if (areaToRegister.virtualizable) {
      initialChildFocusIndex =
        this.virtualizedAreaChildFocusIndices[areaToRegister.focusId] ?? 0;
      delete this.virtualizedAreaChildFocusIndices[areaToRegister.focusId];
    }

    // Ensure that the starting child focus is in bounds for the area if replacing
    // a previous area that might have had a different element count.
    areaToRegister.childFocusIndex = Math.min(
      initialChildFocusIndex,
      areaToRegister.elementCount - 1,
    );

    this.focusAreas[areaToRegister.focusId] = areaToRegister;

    const parentIsFocused = focusIdContainsElement({
      elementId: parentFocusId,
      focusId: this.currentFocusId,
    });

    if (parentIsFocused) {
      const parentChildFocusIndex =
        parentFocusId === ROOT_FOCUS_ID
          ? 0
          : this.getAreaElement(parentFocusId).childFocusIndex;

      if (parentChildFocusIndex === areaToRegister.focusIndex) {
        const deepestFocusId = this.findDeepestFocusId(areaToRegister.focusId);

        // Ensure that focus is propagated down to deepest possible element. This needs to happen:
        // 1) During initial render as navigation areas are first being created and registered.
        // 2) When a newly created nav area takes focus on first render.
        // 3) A navigation area is replaced with an area containing a different element count.
        if (deepestFocusId !== this.currentFocusId) {
          if (areaToReplace) {
            // Children have already been created and need to be notified of a focus change that might affect them.
            this.focusElement(deepestFocusId);
          } else {
            // Children have not yet been created. They will read this value as part of mounting to establish initial
            // focus state, no need to broadcast a focus change.
            this.currentFocusId = deepestFocusId;
          }
        }
      }
    }
  };

  public removeArea = (areaToRemove: FocusableAreaElement): void => {
    const storedArea = this.focusAreas[areaToRemove.focusId];

    // Bail if the area to remove has already been replaced by a new area.
    // The new area will always register & replace the old area before the old
    // area runs its self-removal logic.
    if (!storedArea || storedArea.instanceId !== areaToRemove.instanceId) {
      return;
    }

    // If the area being removed didn't have a replacement and it is in the focus
    // path, it was replaced by a leaf, update focus accordingly
    if (
      focusIdContainsElement({
        elementId: areaToRemove.focusId,
        focusId: this.currentFocusId,
      })
    ) {
      // No need to broadcast this change, the focus tree is still correct
      this.currentFocusId = areaToRemove.focusId;
    }

    if (areaToRemove.virtualizable) {
      this.virtualizedAreaChildFocusIndices[areaToRemove.focusId] =
        areaToRemove.childFocusIndex;
    } else {
      this.clearVirtualizedFocusIndices(areaToRemove.focusId);
    }

    delete this.focusAreas[areaToRemove.focusId];
  };

  public broadcastFocusChange = (): void => {
    // Rather than directly sending state changes to the listeners, they
    // query FocusBroadcaster directly once they are notified of a state change.
    // This minimizes the risk of race-conditions, need to constantly re-register listeners, etc.
    Object.values(this.focusChangeListeners).forEach((listener) => listener());
  };

  /**
   * Adds a listener that is invoked any time focus is changed.
   */
  public addFocusChangeListener = (listener: () => void): (() => void) => {
    const key = this.listenerIncrement;
    this.listenerIncrement++;

    this.focusChangeListeners[key] = listener;

    return (): void => {
      delete this.focusChangeListeners[key];
    };
  };

  public getCurrentFocusId(): string {
    return this.currentFocusId;
  }

  private clearVirtualizedFocusIndices(rootFocusId: string): void {
    Object.entries(this.virtualizedAreaChildFocusIndices).forEach(
      ([virtualizedAreaFocusId]) => {
        if (
          focusIdContainsElement({
            elementId: rootFocusId,
            focusId: virtualizedAreaFocusId,
          })
        ) {
          delete this.virtualizedAreaChildFocusIndices[virtualizedAreaFocusId];
        }
      },
    );
  }

  private findDeepestFocusId(focusId: string): string {
    const matchingArea = this.focusAreas[focusId];
    if (matchingArea) {
      return this.findDeepestFocusId(
        getFocusId(matchingArea.focusId, matchingArea.childFocusIndex),
      );
    }

    const { childFocusIndex, parentFocusId } = splitFocusId(focusId);
    const parentFocusArea = this.focusAreas[parentFocusId];
    if (!parentFocusArea) {
      throw new Error(`No area found when finding deepest for: ${focusId}`);
    }

    if (
      childFocusIndex < 0 ||
      childFocusIndex >= parentFocusArea.elementCount
    ) {
      throw new Error(`Focus ID is out of bounds: ${focusId}`);
    }

    return focusId;
  }
}
