import { mockArea } from '../../test-mocks';
import { FocusBroadcaster, ROOT_FOCUS_ID, getFocusId } from '.';

describe(FocusBroadcaster, () => {
  function setup() {
    const broadcaster = new FocusBroadcaster();

    const rootArea = mockArea({
      childFocusIndex: 0,
      elementCount: 1,
      focusIndex: 0,
      parentFocusId: ROOT_FOCUS_ID,
    });

    broadcaster.registerArea(rootArea);

    return { broadcaster, rootArea };
  }

  describe('area registration', () => {
    it('sets current focus to the default focused child of the root area', () => {
      const { broadcaster, rootArea } = setup();
      expect(
        broadcaster.isFocused(rootArea.parentFocusId, rootArea.focusIndex),
      ).toEqual(true);
      expect(broadcaster.getAreaElement(rootArea.focusId)).toEqual(rootArea);

      // focus should be forwarded to the rootArea's childFocusIndex
      expect(
        broadcaster.isFocused(rootArea.focusId, rootArea.childFocusIndex),
      ).toEqual(true);
    });

    it('sets current focus to the default focused child of the area that is focused', () => {
      const { broadcaster, rootArea } = setup();

      const area = mockArea({
        elementCount: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(area);
      expect(broadcaster.getAreaElement(area.focusId)).toEqual(area);
      expect(
        broadcaster.isFocused(area.parentFocusId, area.focusIndex),
      ).toEqual(true);
      // focus should be forwarded to the area's childFocusIndex
      expect(broadcaster.isFocused(area.focusId, area.childFocusIndex)).toEqual(
        true,
      );
    });

    it('does not set the focus Id to a child not matching the focus index of the deepest focused parent', () => {
      const { broadcaster, rootArea } = setup();

      const area = mockArea({
        elementCount: 1,
        focusIndex: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(area);

      expect(
        broadcaster.isFocused(area.parentFocusId, area.focusIndex),
      ).toEqual(false);

      // focus should be forwarded to the area's childFocusIndex
      expect(broadcaster.isFocused(area.focusId, area.childFocusIndex)).toEqual(
        false,
      );
    });

    it('preserves the childFocusIndex of a virtualized area being re-registered', () => {
      const { broadcaster, rootArea } = setup();

      const initialArea = mockArea({
        childFocusIndex: 2,
        elementCount: 3,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
        virtualizable: true,
      });

      broadcaster.registerArea(initialArea);
      broadcaster.removeArea(initialArea);

      const replacementArea = mockArea({
        childFocusIndex: 0,
        elementCount: 3,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
        virtualizable: true,
      });
      broadcaster.registerArea(replacementArea);

      expect(
        broadcaster.getAreaElement(replacementArea.focusId).childFocusIndex,
      ).toEqual(initialArea.childFocusIndex);
    });

    it('clamps the childFocusIndex of an area being re-registered', () => {
      const { broadcaster, rootArea } = setup();

      const initialArea = mockArea({
        childFocusIndex: 2,
        elementCount: 3,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
      });

      broadcaster.registerArea(initialArea);
      broadcaster.removeArea(initialArea);

      const replacementArea = mockArea({
        childFocusIndex: 0,
        elementCount: 1,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(replacementArea);

      expect(
        broadcaster.getAreaElement(replacementArea.focusId).childFocusIndex,
      ).toEqual(0);
    });

    it('preserves the childFocusIndex of an area being replaced by an area of the same type', () => {
      const { broadcaster, rootArea } = setup();

      const initialArea = mockArea({
        childFocusIndex: 2,
        elementCount: 3,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
        type: 'vertical',
      });
      broadcaster.registerArea(initialArea);

      const replacementArea = mockArea({
        childFocusIndex: 0,
        elementCount: 3,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
        type: 'vertical',
      });
      broadcaster.registerArea(replacementArea);

      expect(
        broadcaster.getAreaElement(replacementArea.focusId).childFocusIndex,
      ).toEqual(initialArea.childFocusIndex);
    });

    it('does not preserve the childFocusIndex of an area being replaced by an area of a different type', () => {
      const { broadcaster, rootArea } = setup();

      const initialArea = mockArea({
        childFocusIndex: 2,
        elementCount: 3,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
        type: 'vertical',
      });
      broadcaster.registerArea(initialArea);

      const replacementArea = mockArea({
        childFocusIndex: 0,
        elementCount: 3,
        focusIndex: 0,
        parentFocusId: rootArea.focusId,
        type: 'horizontal',
      });
      broadcaster.registerArea(replacementArea);

      expect(
        broadcaster.getAreaElement(replacementArea.focusId).childFocusIndex,
      ).toEqual(replacementArea.childFocusIndex);
    });
  });

  describe('focusing', () => {
    function focusSetup() {
      const broadcaster = new FocusBroadcaster();

      const rootArea = mockArea({
        childFocusIndex: 0,
        elementCount: 2,
        focusIndex: 0,
        parentFocusId: ROOT_FOCUS_ID,
      });

      broadcaster.registerArea(rootArea);

      const focusChangeListener = jest.fn();
      broadcaster.addFocusChangeListener(focusChangeListener);

      const focusedArea = mockArea({
        elementCount: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(focusedArea);

      const unfocusedArea = mockArea({
        elementCount: 1,
        focusIndex: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(unfocusedArea);

      return {
        broadcaster,
        focusChangeListener,
        focusedArea,
        rootArea,
        unfocusedArea,
      };
    }

    it('updates the focus index and intermediate areas for the new focus ID', () => {
      const {
        broadcaster,
        focusChangeListener,
        focusedArea,
        rootArea,
        unfocusedArea,
      } = focusSetup();
      expect(broadcaster.isFocused(rootArea.focusId, 0)).toEqual(true);
      expect(broadcaster.isFocused(focusedArea.focusId, 0)).toEqual(true);

      expect(focusChangeListener).not.toHaveBeenCalled();
      broadcaster.focusElement(
        getFocusId(unfocusedArea.focusId, unfocusedArea.childFocusIndex),
      );

      expect(
        broadcaster.isFocused(
          unfocusedArea.focusId,
          unfocusedArea.childFocusIndex,
        ),
      ).toEqual(true);
      expect(
        broadcaster.isFocused(focusedArea.focusId, focusedArea.childFocusIndex),
      ).toEqual(false);
      expect(broadcaster.isFocused(rootArea.focusId, 1)).toEqual(true);
      expect(
        broadcaster.getAreaElement(rootArea.focusId).childFocusIndex,
      ).toEqual(1);
      expect(focusChangeListener).toHaveBeenCalledTimes(1);
    });

    it('focuses the deepest leaf when attempting to focus a parent', () => {
      const { broadcaster, unfocusedArea } = focusSetup();
      expect(
        broadcaster.isFocused(
          unfocusedArea.parentFocusId,
          unfocusedArea.focusIndex,
        ),
      ).toEqual(false);

      broadcaster.focusElement(unfocusedArea.focusId);
      expect(
        broadcaster.isFocused(
          unfocusedArea.parentFocusId,
          unfocusedArea.focusIndex,
        ),
      ).toEqual(true);
      expect(
        broadcaster.isFocused(
          unfocusedArea.focusId,
          unfocusedArea.childFocusIndex,
        ),
      ).toEqual(true);
    });
  });

  describe('area removal', () => {
    it('removes an area', () => {
      const { broadcaster, rootArea } = setup();

      const area = mockArea({
        childFocusIndex: 1,
        elementCount: 2,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(area);
      expect(broadcaster.getAreaElement(area.focusId)).toBeTruthy();

      broadcaster.removeArea(area);
      expect(() => broadcaster.getAreaElement(area.focusId)).toThrow();
    });

    it('no-ops if the area to remove has already been replaced', () => {
      const { broadcaster, rootArea } = setup();

      const staleArea = mockArea({
        elementCount: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(staleArea);
      expect(broadcaster.getAreaElement(staleArea.focusId)).toBeTruthy();

      const replacementArea = mockArea({
        elementCount: 2,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(replacementArea);

      broadcaster.removeArea(staleArea);
      expect(broadcaster.getAreaElement(staleArea.focusId)).toEqual(
        replacementArea,
      );
      expect(broadcaster.getAreaElement(replacementArea.focusId)).toEqual(
        replacementArea,
      );
    });

    it('does not adjust current focus ID if a non-focused area is removed', () => {
      const { broadcaster, rootArea } = setup();

      const area = mockArea({
        elementCount: 2,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(area);

      const childArea1 = mockArea({
        elementCount: 1,
        focusIndex: 0,
        parentFocusId: area.focusId,
      });
      broadcaster.registerArea(childArea1);

      const childAreaToRemove = mockArea({
        elementCount: 1,
        focusIndex: 1,
        parentFocusId: area.focusId,
      });
      broadcaster.registerArea(childAreaToRemove);

      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(childArea1.focusId, 0),
      );

      broadcaster.removeArea(childAreaToRemove);
      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(childArea1.focusId, 0),
      );
    });

    it('adjusts the current focus ID if an intermediate focused area is removed', () => {
      const { broadcaster, rootArea } = setup();

      const areaToRemove = mockArea({
        elementCount: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(areaToRemove);

      const childArea = mockArea({
        elementCount: 1,
        parentFocusId: areaToRemove.focusId,
      });
      broadcaster.registerArea(childArea);

      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(childArea.focusId, 0),
      );

      broadcaster.removeArea(areaToRemove);
      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(rootArea.focusId, rootArea.childFocusIndex),
      );
    });

    it('adjusts the current focus ID if the deepest focused area is removed', () => {
      const { broadcaster, rootArea } = setup();

      const areaToRemove = mockArea({
        elementCount: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(areaToRemove);

      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(areaToRemove.focusId, 0),
      );

      broadcaster.removeArea(areaToRemove);
      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(rootArea.focusId, rootArea.childFocusIndex),
      );
    });

    it('does not adjust the current focus ID if the removed focus area is the only one below root', () => {
      const { broadcaster, rootArea } = setup();

      const areaToRemove = mockArea({
        elementCount: 1,
        parentFocusId: rootArea.focusId,
      });
      broadcaster.registerArea(areaToRemove);

      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(areaToRemove.focusId, 0),
      );

      broadcaster.removeArea(areaToRemove);
      expect(broadcaster.getCurrentFocusId()).toEqual(
        getFocusId(rootArea.focusId, 0),
      );
    });
  });
});
