const FOCUS_ID_SEPARATOR = '.';

export const ROOT_FOCUS_ID = 'ROOT';

export function getFocusId(parentId: string, focusIndex: number): string {
  return `${parentId}${FOCUS_ID_SEPARATOR}${focusIndex}`;
}

/**
 * Returns true if the element ID is the active leaf OR a parent area of
 * that leaf.
 */
export function focusIdContainsElement({
  elementId,
  focusId,
}: {
  elementId: string;
  focusId: string;
}): boolean {
  if (focusId === elementId) {
    return true;
  }

  const match = focusId.startsWith(elementId);
  if (!match) {
    return false;
  }

  // Check that the separator is the next character and not an additional digit
  // to prevent accidental matches on (0.1).0 and (0.1)0.0
  return focusId.charAt(elementId.length) === FOCUS_ID_SEPARATOR;
}

/**
 * Returns the child focus index for an area focus ID as defined by a Focus ID.
 */
export function childFocusIndexForFocusedArea(
  focusId: string,
  areaFocusId: string,
): number {
  if (!focusIdContainsElement({ elementId: areaFocusId, focusId })) {
    throw new Error(`Area ${areaFocusId} is not focused for ${focusId}`);
  }

  const focusParts = focusId.split(FOCUS_ID_SEPARATOR);
  const elementParts = areaFocusId.split(FOCUS_ID_SEPARATOR);

  if (elementParts.length === focusParts.length) {
    throw new Error(
      `Element id: ${areaFocusId} is not a parent of focus id: ${focusId}`,
    );
  }

  return Number.parseInt(focusParts[elementParts.length]);
}

export function splitFocusId(focusId: string): {
  childFocusIndex: number;
  parentFocusId: string;
} {
  const lastSeparator = focusId.lastIndexOf(FOCUS_ID_SEPARATOR);
  if (lastSeparator === -1) {
    return { childFocusIndex: 0, parentFocusId: ROOT_FOCUS_ID };
  }

  return {
    childFocusIndex: Number.parseInt(focusId.substr(lastSeparator + 1)),
    parentFocusId: focusId.substr(0, lastSeparator),
  };
}
