import type {
  FocusBroadcaster,
  FocusableAreaElement,
} from '../FocusBroadcaster';
import { ROOT_FOCUS_ID, getFocusId } from '../FocusBroadcaster';
import type { NavInputHandler } from '../useNavInputListener';
import { getIndexForPageNext, getIndexForPagePrev } from './getIndexForPageNav';

// Handles testing nav areas for input handling and then bubbles event up the
// nav tree as necessary until it finds an area to handle the current input.
// The root focus area can only have a single element so we never bubble events
// to it and instead short-circuit for efficiency.
export class InputHandler implements NavInputHandler {
  private broadcaster: FocusBroadcaster;

  constructor(broadcaster: FocusBroadcaster) {
    this.broadcaster = broadcaster;
  }

  public onDown = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (focusedArea.onDown?.()) {
      return;
    }
    if (this.moveToNextElement(focusedArea, focusedArea.verticalIncrement)) {
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onDown(this.broadcaster.getAreaElement(focusedArea.parentFocusId));
    }
  };
  public onRight = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (focusedArea.onRight?.()) {
      return;
    }
    if (this.moveToNextElement(focusedArea, focusedArea.horizontalIncrement)) {
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onRight(this.broadcaster.getAreaElement(focusedArea.parentFocusId));
    }
  };
  public onWheelDown = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (
      focusedArea.handleWheel &&
      this.moveToNextElement(focusedArea, focusedArea.verticalIncrement)
    ) {
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onWheelDown(
        this.broadcaster.getAreaElement(focusedArea.parentFocusId),
      );
    }
  };

  // up, left, and wheelUp all move to the previous element in their
  // respective direction
  public onUp = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (focusedArea.onUp?.()) {
      return;
    }
    if (
      this.moveToPreviousElement(focusedArea, focusedArea.verticalIncrement)
    ) {
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onUp(this.broadcaster.getAreaElement(focusedArea.parentFocusId));
    }
  };
  public onLeft = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (focusedArea.onLeft?.()) {
      return;
    }
    if (
      this.moveToPreviousElement(focusedArea, focusedArea.horizontalIncrement)
    ) {
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onLeft(this.broadcaster.getAreaElement(focusedArea.parentFocusId));
    }
  };
  public onWheelUp = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (
      focusedArea.handleWheel &&
      this.moveToPreviousElement(focusedArea, focusedArea.verticalIncrement)
    ) {
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onWheelUp(
        this.broadcaster.getAreaElement(focusedArea.parentFocusId),
      );
    }
  };
  public onPagePrev = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (focusedArea.pageSize) {
      const newChildFocusIndex = getIndexForPagePrev({
        elementCount: focusedArea.elementCount,
        focusIndex: focusedArea.childFocusIndex,
        pageSize: focusedArea.pageSize,
      });

      this.broadcaster.focusElement(
        getFocusId(focusedArea.focusId, newChildFocusIndex),
      );
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onPagePrev(
        this.broadcaster.getAreaElement(focusedArea.parentFocusId),
      );
    }
  };

  public onPageNext = (
    focusedArea: FocusableAreaElement = this.broadcaster.getFocusedAreaElement(),
  ): void => {
    if (focusedArea.pageSize) {
      const newChildFocusIndex = getIndexForPageNext({
        elementCount: focusedArea.elementCount,
        focusIndex: focusedArea.childFocusIndex,
        pageSize: focusedArea.pageSize,
      });

      this.broadcaster.focusElement(
        getFocusId(focusedArea.focusId, newChildFocusIndex),
      );
      return;
    }

    if (focusedArea.parentFocusId !== ROOT_FOCUS_ID) {
      this.onPageNext(
        this.broadcaster.getAreaElement(focusedArea.parentFocusId),
      );
    }
  };

  private moveToNextElement = (
    focusedArea: FocusableAreaElement,
    increment: number,
  ): boolean => {
    const newChildFocusIndex = focusedArea.childFocusIndex + increment;

    if (newChildFocusIndex < focusedArea.elementCount) {
      this.broadcaster.focusElement(
        getFocusId(focusedArea.focusId, newChildFocusIndex),
      );
      return true;
    }

    return false;
  };
  private moveToPreviousElement = (
    focusedArea: FocusableAreaElement,
    decrement: number,
  ): boolean => {
    const newChildFocusIndex = focusedArea.childFocusIndex - decrement;

    if (newChildFocusIndex >= 0) {
      this.broadcaster.focusElement(
        getFocusId(focusedArea.focusId, newChildFocusIndex),
      );
      return true;
    }

    return false;
  };
}
