import { mockArea } from '../../test-mocks';
import { FocusBroadcaster, getFocusId } from '../FocusBroadcaster';
import { InputHandler } from '.';

describe(InputHandler, () => {
  function setup(
    focusedParent1Overrides?: Partial<Parameters<typeof mockArea>[0]>,
  ) {
    const broadcaster = new FocusBroadcaster();
    const focusElementSpy = jest.spyOn(broadcaster, 'focusElement');
    const inputHandler = new InputHandler(broadcaster);

    const rootArea = mockArea({
      childFocusIndex: 1,
      elementCount: 3,
      handleWheel: true,
      horizontalIncrement: 1,
      pageSize: 1,
      verticalIncrement: 1,
    });
    broadcaster.registerArea(rootArea);

    const unfocusedParent0 = mockArea({
      childFocusIndex: 0,
      elementCount: 1,
      focusIndex: 0,
      horizontalIncrement: 1,
      pageSize: 1,
      parentFocusId: rootArea.focusId,
      verticalIncrement: 1,
    });
    broadcaster.registerArea(unfocusedParent0);

    const focusedParent1 = mockArea({
      childFocusIndex: 0,
      elementCount: 1,
      focusIndex: 1,
      handleWheel: true,
      horizontalIncrement: 1,
      parentFocusId: rootArea.focusId,
      verticalIncrement: 1,
      ...focusedParent1Overrides,
    });
    broadcaster.registerArea(focusedParent1);

    const unfocusedParent2 = mockArea({
      childFocusIndex: 0,
      elementCount: 1,
      focusIndex: 2,
      horizontalIncrement: 1,
      parentFocusId: rootArea.focusId,
      verticalIncrement: 1,
    });
    broadcaster.registerArea(unfocusedParent2);

    // We specifically add this child to focusedParent1 to get
    // additional code coverage in the onPrevPage and onNextPage
    // methods since the child will be evaluated first and has
    // no pageSize.
    const focusedChild = mockArea({
      childFocusIndex: 0,
      elementCount: 1,
      focusIndex: 1,
      horizontalIncrement: 1,
      parentFocusId: focusedParent1.focusId,
      verticalIncrement: 1,
    });
    broadcaster.registerArea(focusedChild);

    return {
      broadcaster,
      focusElementSpy,
      focusedParent1,
      inputHandler,
      rootArea,
      unfocusedParent0,
      unfocusedParent2,
    };
  }

  const mockClientHandler = jest.fn();

  beforeEach(() => {
    mockClientHandler.mockReset();
  });

  describe('down', () => {
    it('increments the area child focus index when in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({ elementCount: 2 });
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onDown();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 1),
      );
    });

    it('increments the areas parent child focus index when index range is exceeded', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup();
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onDown();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 2),
      );
    });

    it('calls the preceding handler before propagating to the nav handler', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup({ onDown: mockClientHandler });
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onDown();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 2),
      );
    });

    it('calls the preceding handler and blocks the nav handler', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({ onDown: mockClientHandler });
      mockClientHandler.mockReturnValueOnce(true);

      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onDown();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).not.toHaveBeenCalled();
    });
  });

  describe('up', () => {
    it('decrements the area child focus index when in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({
          childFocusIndex: 1,
          elementCount: 2,
        });
      expect(broadcaster.isFocused(focusedParent1.focusId, 1)).toEqual(true);
      inputHandler.onUp();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 0),
      );
    });

    it('decrements the areas parent child focus index when index range is exceeded', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup();
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onUp();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 0),
      );
    });

    it('calls the preceding handler before propagating to the nav handler', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup({ onUp: mockClientHandler });
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onUp();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 0),
      );
    });

    it('calls the preceding handler and blocks the nav handler', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({ onUp: mockClientHandler });
      mockClientHandler.mockReturnValueOnce(true);

      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onUp();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).not.toHaveBeenCalled();
    });
  });

  describe('left', () => {
    it('decrements the area child focus index when in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({
          childFocusIndex: 1,
          elementCount: 2,
        });
      expect(broadcaster.isFocused(focusedParent1.focusId, 1)).toEqual(true);
      inputHandler.onLeft();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 0),
      );
    });

    it('decrements the areas parent child focus index when index range is exceeded', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup();
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onLeft();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 0),
      );
    });

    it('calls the preceding handler before propagating to the nav handler', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup({ onLeft: mockClientHandler });
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onLeft();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 0),
      );
    });

    it('calls the preceding handler and blocks the nav handler', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({ onLeft: mockClientHandler });
      mockClientHandler.mockReturnValueOnce(true);

      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onLeft();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).not.toHaveBeenCalled();
    });
  });

  describe('right', () => {
    it('handles right input when the new child index is in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({ elementCount: 2 });
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onRight();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 1),
      );
    });

    it('increments the areas parent child focus index when index range is exceeded', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup();
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onRight();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 2),
      );
    });

    it('calls the preceding handler before propagating to the nav handler', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup({ onRight: mockClientHandler });
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onRight();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 2),
      );
    });

    it('calls the preceding handler and blocks the nav handler', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({ onRight: mockClientHandler });
      mockClientHandler.mockReturnValueOnce(true);

      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onRight();

      expect(mockClientHandler).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).not.toHaveBeenCalled();
    });
  });

  describe('page prev', () => {
    it('decrements the area child focus index when in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({
          childFocusIndex: 1,
          elementCount: 2,
          pageSize: 1,
        });
      expect(broadcaster.isFocused(focusedParent1.focusId, 1)).toEqual(true);
      inputHandler.onPagePrev();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 0),
      );
    });
  });

  describe('page next', () => {
    it('increments the area child focus index when in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({
          childFocusIndex: 1,
          elementCount: 3,
          pageSize: 1,
        });
      expect(broadcaster.isFocused(focusedParent1.focusId, 1)).toEqual(true);
      inputHandler.onPageNext();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 2),
      );
    });
  });

  describe('wheel down', () => {
    it('increments the area child focus index when in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({ elementCount: 2 });
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onWheelDown();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 1),
      );
    });

    it('increments the areas parent child focus index when index range is exceeded', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup();
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onWheelDown();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 2),
      );
    });
  });

  describe('wheel up', () => {
    it('decrements the area child focus index when in bounds', () => {
      const { broadcaster, focusElementSpy, focusedParent1, inputHandler } =
        setup({
          childFocusIndex: 1,
          elementCount: 2,
        });
      expect(broadcaster.isFocused(focusedParent1.focusId, 1)).toEqual(true);
      inputHandler.onWheelUp();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(focusedParent1.focusId, 0),
      );
    });

    it('decrements the areas parent child focus index when index range is exceeded', () => {
      const {
        broadcaster,
        focusElementSpy,
        focusedParent1,
        inputHandler,
        rootArea,
      } = setup();
      expect(broadcaster.isFocused(focusedParent1.focusId, 0)).toEqual(true);
      inputHandler.onWheelUp();

      expect(focusElementSpy).toHaveBeenCalledTimes(1);
      expect(focusElementSpy).toHaveBeenCalledWith(
        getFocusId(rootArea.focusId, 0),
      );
    });
  });
});
