import type { FC } from 'react';
import { createContext, useContext, useEffect, useMemo } from 'react';
import { usePrevious } from 'tachyon-utils-react';
import type {
  FocusBroadcasterMethods,
  FocusableAreaElement,
} from './FocusBroadcaster';
import { FocusBroadcaster, ROOT_FOCUS_ID } from './FocusBroadcaster';
import { InputHandler } from './InputHandler';
import { useNavInputListener } from './useNavInputListener';

export { getFocusId, ROOT_FOCUS_ID } from './FocusBroadcaster';
export type {
  FocusableAreaElement,
  FocusBroadcasterMethods,
} from './FocusBroadcaster';

export type NavigationContext = {
  broadcaster: FocusBroadcasterMethods;
  parentFocusId: string;
};

// istanbul ignore next: trivial
export const navigationContext = createContext<NavigationContext>({
  broadcaster: {
    addFocusChangeListener: () => (): void => {
      return;
    },
    focusElement: (): void => undefined,
    getAreaElement: (): FocusableAreaElement => {
      throw new Error('Cannot use on default context');
    },
    isFocused: (): boolean => false,
    registerArea: (): void => {
      return;
    },
    removeArea: (): void => {
      return;
    },
  },
  parentFocusId: ROOT_FOCUS_ID,
});

// istanbul ignore next: trivial
export function useNavigationContext(): NavigationContext {
  return useContext(navigationContext);
}

export type NavigationRootProps = {
  /**
   * A unique seed for the focus hierarchy.
   */
  focusSeed: string;
  /**
   * The amount of time to throttle subsequent key down inputs in the same direction by in milliseconds.
   */
  keyNavThrottleMs: number;
  /**
   * The amount of time to throttle subsequent wheel inputs by in milliseconds.
   */
  scrollNavThrottleMs: number;
};

/**
 * Render above all other navigable UI in the component hierarchy.
 */
// istanbul ignore next: trivial
export const NavigationRoot: FC<NavigationRootProps> = ({
  children,
  focusSeed,
  keyNavThrottleMs,
  scrollNavThrottleMs,
}) => {
  const prevFocusSeed = usePrevious(focusSeed);

  // Since the focus seed is changing, that means new elements are going to
  // be registering as a part of first render so we cannot wait for the effect
  // to fire or they'll render with the old focus broadcaster
  const ctx = useMemo(() => {
    const broadcaster = new FocusBroadcaster();

    return {
      broadcaster,
      inputHandler: new InputHandler(broadcaster),
      parentFocusId: ROOT_FOCUS_ID,
      seed: focusSeed,
    };
  }, [focusSeed]);

  useNavInputListener({
    inputHandler: ctx.inputHandler,
    keyNavThrottleMs,
    scrollNavThrottleMs,
  });

  useEffect(() => {
    // Elements that are mounting for the first time will have the correct
    // focus values, but elements that are updating won't have known that focus has
    // changed as we build up the new focus hierarchy. We do this in an effect
    // since we know by this point all rendering / registration has been done to
    // limit re-renders / changing state in the middle of expensive work
    if (focusSeed !== prevFocusSeed) {
      ctx.broadcaster.broadcastFocusChange();
    }
  }, [ctx.broadcaster, focusSeed, prevFocusSeed]);

  return <navigationContext.Provider children={children} value={ctx} />;
};

NavigationRoot.displayName = 'NavigationRoot';
