import { useEffect } from 'react';
import { useUserIntent } from 'tachyon-user-intent';
import { throttle } from 'tachyon-utils-stdlib';

export type NavInputHandler = {
  onDown: () => void;
  onLeft: () => void;
  onPageNext: () => void;
  onPagePrev: () => void;
  onRight: () => void;
  onUp: () => void;
  onWheelDown: () => void;
  onWheelUp: () => void;
};

export type UseNavInputListenerProps = {
  inputHandler: NavInputHandler;
  keyNavThrottleMs: number;
  scrollNavThrottleMs: number;
};

/**
 * Responsible for listening for user navigation inputs and triggering directional
 * actions on the provided navigation area ref.
 */
// istanbul ignore next: trivial
export function useNavInputListener({
  inputHandler,
  keyNavThrottleMs,
  scrollNavThrottleMs,
}: UseNavInputListenerProps): void {
  const { mapToIntents } = useUserIntent();

  useEffect(() => {
    const keyThrottle = (callback: () => void) =>
      throttle(callback, keyNavThrottleMs, { leading: true });
    const scrollThrottle = (callback: () => void) =>
      throttle(callback, scrollNavThrottleMs, { leading: true });

    return mapToIntents([
      { callback: keyThrottle(inputHandler.onUp), intent: 'up' },
      { callback: keyThrottle(inputHandler.onDown), intent: 'down' },
      { callback: keyThrottle(inputHandler.onLeft), intent: 'left' },
      { callback: keyThrottle(inputHandler.onRight), intent: 'right' },
      {
        callback: scrollThrottle(inputHandler.onPageNext),
        intent: 'page-right',
      },
      {
        callback: scrollThrottle(inputHandler.onPagePrev),
        intent: 'page-left',
      },
      { callback: scrollThrottle(inputHandler.onWheelUp), intent: 'page-up' },
      {
        callback: scrollThrottle(inputHandler.onWheelDown),
        intent: 'page-down',
      },
    ]);
  }, [inputHandler, keyNavThrottleMs, mapToIntents, scrollNavThrottleMs]);
}
