import type { FC } from 'react';
import { useMemo } from 'react';
import { useEffectOnce } from 'tachyon-utils-react';
import { navigationContext } from '../../NavigationRoot';
import type { AreaArgs } from '../useFocusArea';
import { useFocusArea } from '../useFocusArea';

export type BaseNavProps = Omit<AreaArgs, 'childFocusIndex' | 'uid'> & {
  initialChildFocusIndex?: number;
};

/**
 * Manages navigation focus between elements in an arbitrary area based on user
 * input using up / down / left / right & optionally wheel interactions. The
 * magnitude of any input is controlled by the verticalIncrement and
 * horizontalIncrement props values, and setting either of them equal to
 * elementCount will disable navigation in that direction (delegating to the
 * parent instead).
 *
 * This component can be recursively nested in other navigation areas, allowing
 * for navigation between parent, child, and sibling areas.
 */
export const BaseNav: FC<BaseNavProps> = ({
  children,
  initialChildFocusIndex = 0,
  pageSize = 0,
  takeFocusOnFirstRender,
  ...areaArgs
}) => {
  const { broadcaster, focusId } = useFocusArea({
    ...areaArgs,
    childFocusIndex: initialChildFocusIndex,
    pageSize,
    takeFocusOnFirstRender,
  });

  const ctx = useMemo(
    () => ({
      broadcaster,
      parentFocusId: focusId,
    }),
    [broadcaster, focusId],
  );

  useEffectOnce(() => {
    if (takeFocusOnFirstRender) {
      broadcaster.focusElement(focusId);
    }
  });

  return <navigationContext.Provider children={children} value={ctx} />;
};

BaseNav.displayName = 'BaseNav';
