import type { FC } from 'react';
import { createMountWrapperFactory } from 'tachyon-test-utils';
import { navigationContext, useNavigationContext } from '../../NavigationRoot';
import { mockFocusBroadcaster } from '../../test-mocks';
import { useFocusArea } from '../useFocusArea';
import { BaseNav } from '.';

jest.mock('../useFocusArea', () => {
  const original = jest.requireActual('../useFocusArea');
  return {
    ...original,
    // basically a pass-through spy without resorting to namespace imports
    useFocusArea: jest.fn(original.useFocusArea),
  };
});
const mockUseFocusArea = useFocusArea as jest.Mock;

const ContextReceiver: FC = () => {
  const ctx = useNavigationContext();
  return <div data-ctx={ctx} />;
};

describe(BaseNav, () => {
  const setup = createMountWrapperFactory(
    BaseNav,
    () => ({
      children: <ContextReceiver />,
      debugLabel: 'test',
      elementCount: 2,
      focusIndex: 0,
      handleWheel: false,
      horizontalIncrement: 2,
      type: 'vertical',
      uid: expect.any(String),
      verticalIncrement: 1,
    }),
    {
      wrappingContexts: () => [
        [
          navigationContext,
          {
            broadcaster: mockFocusBroadcaster(),
            parentFocusId: '0',
          },
        ],
      ],
    },
  );

  describe('registration', () => {
    it('sets the received focusId as the parentFocusId for children', () => {
      const { wrapper } = setup();
      expect(wrapper.find('div')).toHaveProp({
        'data-ctx': expect.objectContaining({
          parentFocusId: '0.0',
        }),
      });
    });

    it('uses defaults', () => {
      const {
        props: { children, initialChildFocusIndex, ...p },
      } = setup();
      expect(mockUseFocusArea).toHaveBeenCalledTimes(1);
      expect(mockUseFocusArea).toHaveBeenCalledWith({
        childFocusIndex: 0,
        pageSize: 0,
        ...p,
      });
    });

    it('respects custom overrides', () => {
      const {
        props: { children, initialChildFocusIndex, ...p },
      } = setup({
        initialChildFocusIndex: 1,
        pageSize: 1,
      });
      expect(mockUseFocusArea).toHaveBeenCalledTimes(1);
      expect(mockUseFocusArea).toHaveBeenCalledWith({
        childFocusIndex: 1,
        pageSize: 1,
        ...p,
      });
    });
  });

  describe('taking focus', () => {
    it('calls focus element when takeFocusOnFirstRender is true', () => {
      const { contexts } = setup({ takeFocusOnFirstRender: true });

      expect(
        contexts.get(navigationContext).broadcaster.focusElement,
      ).toHaveBeenCalled();
    });

    it('does not call focus element when takeFocusOnFirstRender is false', () => {
      const { contexts } = setup({ takeFocusOnFirstRender: false });

      expect(
        contexts.get(navigationContext).broadcaster.focusElement,
      ).not.toHaveBeenCalled();
    });
  });
});
