import { useContext, useMemo, useRef } from 'react';
import { useUnmount } from 'tachyon-utils-react';
import { useSubscription } from 'use-subscription';
import { navigationContext } from '../../NavigationRoot';

/**
 * Returns the index of the leaf element that is, or will be, focused for the nav area.
 */
export function useAreaChildFocusIndex(): number {
  const { broadcaster, parentFocusId } = useContext(navigationContext);
  const unmountingRef = useRef(false);

  // Memoize to avoid removing and re-adding subscriptions each time this hook is called.
  const subscription = useMemo(
    () => ({
      getCurrentValue: () =>
        broadcaster.getAreaElement(parentFocusId).childFocusIndex,
      subscribe: (callback: () => void) => {
        const unsubscribe = broadcaster.addFocusChangeListener(() => {
          if (!unmountingRef.current) {
            callback();
          }
        });

        return unsubscribe;
      },
    }),

    // Re-subscribe any time our input changes
    // (e.g. we get a new HTMLInputElement prop to subscribe to)
    [broadcaster, parentFocusId],
  );

  // The value returned by this hook reflects the input's current value.
  // Our component will automatically be re-rendered when that value changes.
  const childFocusIdx = useSubscription(subscription);

  useUnmount(() => {
    unmountingRef.current = true;
  });

  return childFocusIdx;
}
