import { useContext, useMemo, useRef } from 'react';
import { useUnmount } from 'tachyon-utils-react';
import { useSubscription } from 'use-subscription';
import { getFocusId, navigationContext } from '../../NavigationRoot';

type UseFocus = {
  focusId: string;
  focused: boolean;
  /**
   * Changes navigation system focus to the current element.
   */
  takeFocus: () => void;
};

/**
 * Manages focus state for individual elements in the component hierarchy
 * that can be focused.
 */
export function useFocus(focusIndex: number): UseFocus {
  const { broadcaster, parentFocusId } = useContext(navigationContext);
  const unmountingRef = useRef(false);

  // Memoize to avoid removing and re-adding subscriptions each time this hook is called.
  const subscription = useMemo(
    () => ({
      getCurrentValue: () => broadcaster.isFocused(parentFocusId, focusIndex),
      subscribe: (callback: () => void) => {
        const unsubscribe = broadcaster.addFocusChangeListener(() => {
          if (!unmountingRef.current) {
            callback();
          }
        });

        return unsubscribe;
      },
    }),

    // Re-subscribe any time our input changes
    // (e.g. we get a new HTMLInputElement prop to subscribe to)
    [broadcaster, parentFocusId, focusIndex],
  );

  const focusId = getFocusId(parentFocusId, focusIndex);
  const focused = useSubscription(subscription);

  useUnmount(() => {
    unmountingRef.current = true;
  });

  return {
    focusId,
    focused,
    takeFocus: () => {
      broadcaster.focusElement(focusId);
    },
  };
}
