import { act, renderHook } from '@testing-library/react-hooks';
import type { Dispatch, FC, SetStateAction } from 'react';
import { useState } from 'react';
import type { NavigationContext } from '../../NavigationRoot';
import {
  ROOT_FOCUS_ID,
  getFocusId,
  navigationContext,
} from '../../NavigationRoot';
import { mockFocusBroadcaster } from '../../test-mocks';
import { useFocus } from '.';

describe(useFocus, () => {
  function setup() {
    const broadcaster = mockFocusBroadcaster();

    const setContext: {
      current: Dispatch<SetStateAction<NavigationContext>>;
    } = {
      current: () => {
        return;
      },
    };

    const Provider: FC = ({ children }) => {
      const [ctx, setContextState] = useState<NavigationContext>({
        broadcaster,
        parentFocusId: ROOT_FOCUS_ID,
      });

      setContext.current = setContextState;

      return (
        <navigationContext.Provider value={ctx}>
          {children}
        </navigationContext.Provider>
      );
    };

    return {
      Provider,
      broadcaster,
      setContext,
    };
  }

  it('reads the initial focus from the broadcaster', () => {
    const { Provider, broadcaster } = setup();
    (broadcaster.isFocused as jest.Mock).mockReturnValue(true);

    const { result } = renderHook(() => useFocus(0), {
      wrapper: Provider,
    });

    expect(result.current.focused).toEqual(true);
  });

  it('registers a focus listener on mount and removes on unmount', () => {
    const { Provider, broadcaster } = setup();
    const { unmount } = renderHook(() => useFocus(0), {
      wrapper: Provider,
    });

    expect(broadcaster.addFocusChangeListener).toHaveBeenCalledTimes(1);
    expect(broadcaster.removeChangeListener).not.toHaveBeenCalled();

    unmount();
    expect(broadcaster.removeChangeListener).toHaveBeenCalledTimes(1);
  });

  it('updates state if the focus ID matches', () => {
    const { Provider, broadcaster } = setup();
    (broadcaster.isFocused as jest.Mock).mockReturnValue(false);

    const { result } = renderHook(() => useFocus(0), {
      wrapper: Provider,
    });
    expect(result.current.focused).toEqual(false);
    expect(broadcaster.addFocusChangeListener).toHaveBeenCalledTimes(1);

    (broadcaster.isFocused as jest.Mock).mockReturnValue(true);
    act(() => {
      (broadcaster.addFocusChangeListener as jest.Mock).mock.calls[0][0]();
    });
    expect(result.current.focused).toEqual(true);

    (broadcaster.isFocused as jest.Mock).mockReturnValue(false);
    act(() => {
      (broadcaster.addFocusChangeListener as jest.Mock).mock.calls[0][0]();
    });
    expect(result.current.focused).toEqual(false);
  });

  it('does not set state if the listener is invoked while unmounting', () => {
    const { Provider, broadcaster } = setup();
    (broadcaster.isFocused as jest.Mock).mockReturnValue(false);

    const { result, unmount } = renderHook(() => useFocus(0), {
      wrapper: Provider,
    });
    expect(result.current.focused).toEqual(false);

    unmount();
    (broadcaster.isFocused as jest.Mock).mockReturnValue(true);
    act(() => {
      (broadcaster.addFocusChangeListener as jest.Mock).mock.calls[0][0]();
    });
    expect(result.current.focused).toEqual(false);
  });

  it('reloads state / listener if broadcaster changes', () => {
    const { Provider, broadcaster, setContext } = setup();
    (broadcaster.isFocused as jest.Mock).mockReturnValue(false);

    const { result, unmount } = renderHook(() => useFocus(0), {
      wrapper: Provider,
    });
    expect(result.current.focused).toEqual(false);
    expect(broadcaster.removeChangeListener).not.toHaveBeenCalled();

    const newBroadcaster = mockFocusBroadcaster();
    (newBroadcaster.isFocused as jest.Mock).mockReturnValue(true);

    act(() => {
      setContext.current({
        broadcaster: newBroadcaster,
        parentFocusId: ROOT_FOCUS_ID,
      });
    });

    expect(result.current.focused).toEqual(true);
    expect(broadcaster.removeChangeListener).toHaveBeenCalledTimes(1);
    expect(newBroadcaster.addFocusChangeListener).toHaveBeenCalledTimes(1);
    expect(newBroadcaster.removeChangeListener).toHaveBeenCalledTimes(0);

    (newBroadcaster.isFocused as jest.Mock).mockReturnValue(false);
    act(() => {
      (newBroadcaster.addFocusChangeListener as jest.Mock).mock.calls[0][0]();
    });
    expect(result.current.focused).toEqual(false);

    unmount();
    expect(newBroadcaster.removeChangeListener).toHaveBeenCalledTimes(1);
  });

  it('reloads state / listener if parentFocusId changes', () => {
    const { Provider, broadcaster, setContext } = setup();
    (broadcaster.isFocused as jest.Mock).mockReturnValue(false);

    const { result, unmount } = renderHook(() => useFocus(0), {
      wrapper: Provider,
    });
    expect(result.current.focused).toEqual(false);
    expect(broadcaster.removeChangeListener).not.toHaveBeenCalled();

    (broadcaster.isFocused as jest.Mock).mockReturnValue(true);

    const newParentFocusId = '0.0';
    act(() => {
      setContext.current({
        broadcaster,
        parentFocusId: newParentFocusId,
      });
    });

    expect(result.current.focused).toEqual(true);
    expect(broadcaster.removeChangeListener).toHaveBeenCalledTimes(1);
    expect(broadcaster.addFocusChangeListener).toHaveBeenCalledTimes(2);

    (broadcaster.isFocused as jest.Mock).mockReturnValue(false);
    act(() => {
      (broadcaster.addFocusChangeListener as jest.Mock).mock.calls[1][0]();
    });
    expect(result.current.focused).toEqual(false);

    unmount();
    expect(broadcaster.removeChangeListener).toHaveBeenCalledTimes(2);
  });

  it('takeFocus calls focusElement on the broadcaster', () => {
    const { Provider, broadcaster } = setup();
    const { result } = renderHook(() => useFocus(0), {
      wrapper: Provider,
    });
    expect(broadcaster.focusElement).not.toHaveBeenCalled();

    result.current.takeFocus();
    expect(broadcaster.focusElement).toHaveBeenCalledWith(
      getFocusId(ROOT_FOCUS_ID, 0),
    );
  });
});
