import { useContext, useMemo, useRef } from 'react';
import { useUnmount } from 'tachyon-utils-react';
import type {
  FocusableAreaElement,
  NavigationContext,
} from '../../NavigationRoot';
import { getFocusId, navigationContext } from '../../NavigationRoot';
import { useFocus } from '../useFocus';

type UseFocusElement = {
  broadcaster: NavigationContext['broadcaster'];
  focusId: string;
  focused: boolean;
};

export type AreaArgs = Omit<
  FocusableAreaElement,
  'focusId' | 'instanceId' | 'parentFocusId'
>;

/**
 * Manages registering the area with the focus system and updating focus state.
 */
export function useFocusArea({
  childFocusIndex,
  debugLabel,
  elementCount,
  focusIndex,
  handleWheel,
  horizontalIncrement,
  onDown,
  onLeft,
  onRight,
  onUp,
  pageSize,
  type,
  verticalIncrement,
  virtualizable,
}: AreaArgs): UseFocusElement {
  const { broadcaster, parentFocusId } = useContext(navigationContext);
  const { current: instanceId } = useRef(Math.random().toString());

  // Memoize the args to avoid triggering the area re-registration
  // effect unnecessarily
  const area = useMemo<FocusableAreaElement>(
    () => ({
      childFocusIndex,
      debugLabel,
      elementCount,
      focusId: getFocusId(parentFocusId, focusIndex),
      focusIndex,
      handleWheel,
      horizontalIncrement,
      instanceId,
      onDown,
      onLeft,
      onRight,
      onUp,
      pageSize,
      parentFocusId,
      type,
      verticalIncrement,
      virtualizable,
    }),
    [
      childFocusIndex,
      debugLabel,
      elementCount,
      focusIndex,
      handleWheel,
      horizontalIncrement,
      instanceId,
      onDown,
      onLeft,
      onRight,
      onUp,
      pageSize,
      parentFocusId,
      virtualizable,
      type,
      verticalIncrement,
    ],
  );

  const { focused } = useFocus(area.focusIndex);

  // Abusing the intended purpose of `useMemo` to ensure that initial and
  // subsequent re-registrations happen only as necessary and always synchronously
  // since children may also attempt to read state synchronously via useSubscription
  useMemo(() => {
    broadcaster.registerArea(area);
  }, [broadcaster, area]);

  // Only remove the area on unmount to avoid race conditions / side-effects.
  // We use these refs to ensure that when it is time remove the area, the values
  // aren't stale
  const broadcasterRef = useRef(broadcaster);
  const areaRef = useRef(area);
  broadcasterRef.current = broadcaster;
  areaRef.current = area;
  useUnmount(() => {
    broadcasterRef.current.removeArea(areaRef.current);
  });

  return useMemo(
    () => ({
      broadcaster,
      focusId: area.focusId,
      focused,
    }),
    [broadcaster, area, focused],
  );
}
