import { act, renderHook } from '@testing-library/react-hooks';
import type { Dispatch, FC, SetStateAction } from 'react';
import { useState } from 'react';
import type { NavigationContext } from '../../NavigationRoot';
import {
  ROOT_FOCUS_ID,
  getFocusId,
  navigationContext,
} from '../../NavigationRoot';
import { mockFocusBroadcaster } from '../../test-mocks';
import type { AreaArgs } from '.';
import { useFocusArea } from '.';

let mockUseFocusValue = false;
jest.mock('../useFocus', () => ({
  useFocus: () => ({ focused: mockUseFocusValue }),
}));

function mockAreaArgs(
  overrides: Partial<AreaArgs> & {
    elementCount: number;
  },
): AreaArgs {
  const focusIndex = overrides.focusIndex ?? 0;

  return {
    childFocusIndex: 0,
    debugLabel: 'mockAreaArgs',
    focusIndex,
    handleWheel: false,
    horizontalIncrement: 10,
    type: 'vertical',
    verticalIncrement: 1,
    ...overrides,
  };
}

describe(useFocusArea, () => {
  beforeEach(() => {
    mockUseFocusValue = false;
  });

  function setup() {
    const broadcaster = mockFocusBroadcaster();
    const parentFocusId = ROOT_FOCUS_ID;

    const setContext: {
      current: Dispatch<SetStateAction<NavigationContext>>;
    } = {
      current: () => {
        return;
      },
    };

    const Provider: FC = ({ children }) => {
      const [ctx, setContextState] = useState<NavigationContext>({
        broadcaster,
        parentFocusId,
      });

      setContext.current = setContextState;

      return (
        <navigationContext.Provider value={ctx}>
          {children}
        </navigationContext.Provider>
      );
    };

    return {
      Provider,
      broadcaster,
      parentFocusId,
      setContext,
    };
  }

  it('registers on initial render and returns expected values', () => {
    const { Provider, broadcaster, parentFocusId } = setup();
    const areaArgs = mockAreaArgs({ elementCount: 1 });
    const { result } = renderHook(() => useFocusArea(areaArgs), {
      wrapper: Provider,
    });

    expect(broadcaster.registerArea).toHaveBeenCalledWith({
      ...areaArgs,
      focusId: getFocusId(parentFocusId, areaArgs.focusIndex),
      instanceId: expect.any(String),
      pageSize: undefined,
      parentFocusId,
      virtualized: undefined,
    });

    expect(result.current).toEqual({
      broadcaster,
      focusId: getFocusId(parentFocusId, areaArgs.focusIndex),
      focused: mockUseFocusValue,
    });
  });

  it('re-registers the area when parameters change', () => {
    const { Provider, broadcaster, parentFocusId } = setup();
    const areaArgs = mockAreaArgs({ elementCount: 1 });
    const { rerender } = renderHook((args: any) => useFocusArea(args), {
      initialProps: areaArgs,
      wrapper: Provider,
    });

    expect(broadcaster.registerArea).toHaveBeenCalledTimes(1);
    expect(broadcaster.registerArea).toHaveBeenCalledWith({
      ...areaArgs,
      focusId: getFocusId(parentFocusId, areaArgs.focusIndex),
      instanceId: expect.any(String),
      pageSize: undefined,
      parentFocusId,
      virtualized: undefined,
    });

    // Ensure memoization is working
    rerender(mockAreaArgs({ elementCount: 1 }));
    expect(broadcaster.registerArea).toHaveBeenCalledTimes(1);

    const secondAreaArgs = mockAreaArgs({ elementCount: 2 });
    rerender(secondAreaArgs);

    expect(broadcaster.registerArea).toHaveBeenCalledTimes(2);
    expect(broadcaster.registerArea).toHaveBeenLastCalledWith({
      ...secondAreaArgs,
      focusId: getFocusId(parentFocusId, areaArgs.focusIndex),
      instanceId: expect.any(String),
      pageSize: undefined,
      parentFocusId,
      virtualized: undefined,
    });
  });

  it('re-registers when the contexts broadcaster changes', () => {
    const { Provider, broadcaster, parentFocusId, setContext } = setup();
    const areaArgs = mockAreaArgs({ elementCount: 1 });
    renderHook((args: any) => useFocusArea(args), {
      initialProps: areaArgs,
      wrapper: Provider,
    });
    expect(broadcaster.registerArea).toHaveBeenCalledTimes(1);

    const newBroadcaster = mockFocusBroadcaster();
    act(() => {
      setContext.current({
        broadcaster: newBroadcaster,
        parentFocusId,
      });
    });

    expect(newBroadcaster.registerArea).toHaveBeenCalledTimes(1);
  });

  it('re-registers when the parentFocusId changes', () => {
    const { Provider, broadcaster, setContext } = setup();
    const areaArgs = mockAreaArgs({ elementCount: 1 });
    renderHook((args: any) => useFocusArea(args), {
      initialProps: areaArgs,
      wrapper: Provider,
    });
    expect(broadcaster.registerArea).toHaveBeenCalledTimes(1);

    const newParentFocusId = '0.0';
    act(() => {
      setContext.current({
        broadcaster,
        parentFocusId: newParentFocusId,
      });
    });

    expect(broadcaster.registerArea).toHaveBeenCalledTimes(2);
    expect(broadcaster.registerArea).toHaveBeenLastCalledWith(
      expect.objectContaining({
        parentFocusId: newParentFocusId,
      }),
    );
  });

  it('unregisters on unmount', () => {
    const { Provider, broadcaster } = setup();
    const { unmount } = renderHook(
      () => useFocusArea(mockAreaArgs({ elementCount: 1 })),
      {
        wrapper: Provider,
      },
    );
    expect(broadcaster.removeArea).not.toHaveBeenCalled();

    unmount();
    expect(broadcaster.removeArea).toHaveBeenCalledTimes(1);
  });
});
