import type { FC } from 'react';
import { createContext, useContext, useRef } from 'react';
import { logger } from 'tachyon-logger';
import { useConst } from 'tachyon-utils-react';
import type { Intent } from '../types';

type UserIntentContext = {
  mapToIntents: MapToIntents;
  subscribeToIntent: SubscribeToIntent;
  triggerIntent: TriggerIntent;
};

type SubscribeToIntentOpts = {
  /**
   * Callback to be executed when the intent is fired
   */
  callback: () => void;
  /**
   * Which intent the callback is listening for
   */
  intent: Intent;
  /**
   * If this is marked true, other listeners will not be fired for this intent.
   * Only one blocking listener is allowed for each intent.
   */
  preventDefault?: true;
};

/**
 * @returns a callback that will remove the listener. This works well with useEffect
 */
type SubscribeToIntent = (
  subscribeToIntentOpts: SubscribeToIntentOpts,
) => () => void;

type TriggerIntent = (intent: Intent) => void;

type MapToIntents = (
  intentListeners: Array<SubscribeToIntentOpts>,
) => () => void;

// istanbul ignore next: trivial
const userIntentContext = createContext<UserIntentContext>({
  mapToIntents: () => () => undefined,
  subscribeToIntent: () => () => undefined,
  triggerIntent: () => undefined,
});

// istanbul ignore next: trivial
/**
 * Hook used for adding listeners to intents as well as triggering intents programatically
 *
 * @returns
 * - {@link} MapToIntents mapToIntents} - sets up multiple intent listeners {@link Intent}
 * - {@link TriggerIntent triggerIntent} - imperative functio to trigger an {@link Intent}
 * - {@link SubscribeToIntent subscribeToIntent} - sets up a subscriber to an {@link Intent}
 */
export function useUserIntent(): UserIntentContext {
  return useContext(userIntentContext);
}

type BlockingListeners = {
  [key in Intent]?: () => void;
};

type Listeners = {
  [key in Intent]?: Set<() => void>;
};

export const UserIntentRoot: FC = ({ children }) => {
  const listeners = useRef<Listeners>({});
  const blockingListeners = useRef<BlockingListeners>({});

  const value = useConst(() => {
    const subscribeToIntent = ({
      callback,
      intent,
      preventDefault,
    }: SubscribeToIntentOpts) => {
      if (preventDefault) {
        if (blockingListeners.current[intent]) {
          logger.warn({
            category: 'UserIntentRoot',
            message: `Attempted registry of multiple blocking listeners on ${intent}`,
            package: 'user-intent',
          });

          return () => undefined;
        }

        blockingListeners.current[intent] = callback;
        return () => {
          blockingListeners.current[intent] = undefined;
        };
      }

      listeners.current[intent] ??= new Set();
      listeners.current[intent]?.add(callback);

      return () => {
        listeners.current[intent]?.delete(callback);
      };
    };

    return {
      mapToIntents: (intentListeners: Array<SubscribeToIntentOpts>) => {
        const listenersToRemove: Array<() => void> = [];
        intentListeners.forEach((subscribeToIntentOpts) => {
          listenersToRemove.push(subscribeToIntent(subscribeToIntentOpts));
        });
        return () => {
          listenersToRemove.forEach((removeListener) => {
            removeListener();
          });
        };
      },
      subscribeToIntent,
      triggerIntent: (intent: Intent) => {
        const blockingCallback = blockingListeners.current[intent];
        if (blockingCallback) {
          blockingCallback();
          return;
        }

        listeners.current[intent]?.forEach((callback) => callback());
      },
    };
  });

  return <userIntentContext.Provider children={children} value={value} />;
};

UserIntentRoot.displayName = 'UserIntentRoot';
