import { act, renderHook } from '@testing-library/react-hooks';
import { UserIntentRoot, useUserIntent } from '.';

jest.mock('tachyon-logger', () => ({ logger: { warn: jest.fn() } }));

describe(useUserIntent, () => {
  const setup = () => renderHook(useUserIntent, { wrapper: UserIntentRoot });
  describe('subscribeToIntent', () => {
    it('adds and removes listeners on intents with subscribeToIntent', () => {
      const { result } = setup();
      const { subscribeToIntent, triggerIntent } = result.current;
      const mockCallback = jest.fn();
      const persistedMockCallback = jest.fn();
      const removeListener = subscribeToIntent({
        callback: mockCallback,
        intent: 'back',
      });
      subscribeToIntent({
        callback: persistedMockCallback,
        intent: 'back',
      });

      expect(persistedMockCallback).not.toHaveBeenCalled();
      expect(mockCallback).not.toHaveBeenCalled();

      triggerIntent('back');

      expect(mockCallback).toHaveBeenCalledTimes(1);
      expect(persistedMockCallback).toHaveBeenCalledTimes(1);

      removeListener();
      triggerIntent('back');

      expect(persistedMockCallback).toHaveBeenCalledTimes(2);
      expect(mockCallback).toHaveBeenCalledTimes(1);
    });

    it('adds and removes listeners on intents with mapToIntents', () => {
      const { result } = setup();
      const { mapToIntents, triggerIntent } = result.current;
      const mockCallback = jest.fn();
      const mockCallback2 = jest.fn();
      const removeListeners = mapToIntents([
        {
          callback: mockCallback,
          intent: 'back',
        },
        {
          callback: mockCallback2,
          intent: 'back',
        },
      ]);

      expect(mockCallback2).not.toHaveBeenCalled();
      expect(mockCallback).not.toHaveBeenCalled();

      triggerIntent('back');

      expect(mockCallback).toHaveBeenCalledTimes(1);
      expect(mockCallback2).toHaveBeenCalledTimes(1);

      removeListeners();
      triggerIntent('back');

      expect(mockCallback2).toHaveBeenCalledTimes(1);
      expect(mockCallback).toHaveBeenCalledTimes(1);
    });

    it('calls a blocking listener and does not dispatch other listeners', () => {
      const { result } = setup();
      const mockListener = jest.fn();
      const mockBlockingListener = jest.fn();
      let unmountBlockingListener: any;

      act(() => {
        unmountBlockingListener = result.current.subscribeToIntent({
          callback: mockBlockingListener,
          intent: 'back',
          preventDefault: true,
        });
      });

      result.current.subscribeToIntent({
        callback: mockListener,
        intent: 'back',
      });
      result.current.triggerIntent('back');

      expect(mockListener).not.toHaveBeenCalled();
      expect(mockBlockingListener).toHaveBeenCalledTimes(1);

      act(() => {
        unmountBlockingListener();
      });
      result.current.triggerIntent('back');

      expect(mockListener).toHaveBeenCalledTimes(1);
      expect(mockBlockingListener).toHaveBeenCalledTimes(1);
    });

    it('adds a blocking listener and disregards subsequent blocking listeners being added', () => {
      const { result } = setup();
      const mockBlockingListener = jest.fn();
      const mockIgnoredBlockingListener = jest.fn();

      act(() => {
        result.current.subscribeToIntent({
          callback: mockBlockingListener,
          intent: 'back',
          preventDefault: true,
        });
        result.current.subscribeToIntent({
          callback: mockBlockingListener,
          intent: 'back',
          preventDefault: true,
        });
      });

      result.current.triggerIntent('back');

      expect(mockIgnoredBlockingListener).not.toHaveBeenCalled();
      expect(mockBlockingListener).toHaveBeenCalledTimes(1);
    });
  });
});
