import type { PulsarBackend } from 'pulsar-utils';

/**
 * Source:
 * https://docs.google.com/document/d/1W6wX_vN30UvkA8dbbXrydyJUJ0k42FUEalWvDj4Aetc/edit#heading=h.lekhvbl853q0
 */
export type SpadeBaseStaticProperties = {
  /**
   * The underlying library use to accomplish playback. Use "native" as the value
   * when using built-in Browser methods.
   */
  backend: PulsarBackend;
};

/**
 * Source:
 * https://docs.google.com/document/d/1W6wX_vN30UvkA8dbbXrydyJUJ0k42FUEalWvDj4Aetc/edit#heading=h.lekhvbl853q0
 *
 * Only include fields that are derivable from an HTMLVideoElement.
 */
export type SpadeBaseDynamicProperties = {
  muted: boolean;
  play_session_id: string;
  vid_height: number;
  vid_width: number;
  volume: number;
};

/**
 * Full Base Event Properties interface. Utilize the "baseEventProperties" to
 * most simply construct this object.
 */
export type SpadeBaseProperties = SpadeBaseDynamicProperties &
  SpadeBaseStaticProperties;

type SpadeDynamicPropertiesArgs = {
  playSessionId: string;
  video: HTMLVideoElement;
};

// istanbul ignore next: trivial
export function dynamicEventProperties({
  playSessionId,
  video,
}: SpadeDynamicPropertiesArgs): SpadeBaseDynamicProperties {
  return {
    muted: video.muted,
    play_session_id: playSessionId,
    vid_height: video.videoHeight,
    vid_width: video.videoWidth,
    volume: video.volume,
  };
}

export type BaseEventPropertiesArgs = SpadeBaseStaticProperties &
  SpadeDynamicPropertiesArgs;

// istanbul ignore next: trivial
/**
 * Convenience function that combines static and dynamic event base tracking
 * properties to limit the number of values the consumer needs to manage.
 */
export function baseEventProperties({
  backend,
  ...dynamicArgs
}: BaseEventPropertiesArgs): SpadeBaseProperties {
  return {
    ...dynamicEventProperties(dynamicArgs),
    backend,
  };
}
