import type { PlayerAnalyticsEvent } from 'pulsar-utils';
import type { SpadeBaseProperties } from './baseProperties';

/**
 * Spade formatted "minute-watched" event.
 *
 * Refer to the tracking spec for field definitions:
 * https://docs.google.com/document/d/1W6wX_vN30UvkA8dbbXrydyJUJ0k42FUEalWvDj4Aetc/edit#
 */
type MinuteWatchedEvent = SpadeBaseProperties & {
  /**
   * Cumulative number of minute watched events that have been sent inclusive
   * of this one.
   */
  minutes_logged: number;
  /**
   * Whether or not this is the first play of this Pulsar instantiation.
   */
  seconds_offset: number;
};

/**
 * Dynamic Minute Watched event properties.
 */
type MinuteWatchedProperties = SpadeBaseProperties & {
  firstMinuteWatchedOffsetMs: number;
  minutesWatched: number;
};

// istanbul ignore next: trivial
/**
 * Generates a Spade Formatted "minute-watched" event.
 */
export function minuteWatchedEvent({
  firstMinuteWatchedOffsetMs,
  minutesWatched,
  ...baseProperties
}: MinuteWatchedProperties): PlayerAnalyticsEvent {
  const properties: MinuteWatchedEvent = {
    ...baseProperties,
    minutes_logged: minutesWatched,
    seconds_offset: firstMinuteWatchedOffsetMs / 1000,
  };

  return {
    name: 'minute-watched',
    properties,
  };
}
