import type { PlayerAnalyticsEvent } from 'pulsar-utils';
import type { SpadeBaseProperties } from './baseProperties';

/**
 * Spade formatted "video_error" event:
 * https://blueprint.di.xarth.tv/#/schema/video_error
 */
type VideoErrorEvent = SpadeBaseProperties & {
  /**
   * Internal error code.
   */
  video_error_code: number;
  /**
   * Message describing the error.
   */
  video_error_message: string;
  /**
   * Whether or not the error is recoverable and playback may continue. In the
   * case of Pulsar, none of our errors are recoverable because we are using
   * Native HLS.
   */
  video_error_recoverable: false;
  /**
   * See MediaResult: https://git-aws.internal.justin.tv/video/player-core/blob/aae4ac37571e5e2aa030e8c01f848a69aceaeda3/src/player/result.cpp.
   * In the case of Pulsar, the value is always "Error" because the native HTML
   * player is responsible for emitting errors limiting the granularity we have
   * into specific error causes.
   */
  video_error_result: 'Error';
  /**
   * See ErrorSource: https://git-aws.internal.justin.tv/video/player-core/blob/aae4ac37571e5e2aa030e8c01f848a69aceaeda3/include/mediaplayer/error.hpp.
   * In the case of Pulsar, the value is always "Unspecified" because the native HTML
   * player is responsible for emitting errors limiting the granularity we have
   * into specific error causes.
   */
  video_error_source: 'Unspecified';
  /**
   * Error code describing the error.
   */
  video_error_value: number;
};

/**
 * Dynamic Video Error event properties.
 */
type VideoErrorProperties = SpadeBaseProperties & {
  error: MediaError;
};

/**
 * Generates a Spade Formatted "video_error" event.
 */
export function videoErrorEvent({
  error,
  ...baseProperties
}: VideoErrorProperties): PlayerAnalyticsEvent {
  const properties: VideoErrorEvent = {
    ...baseProperties,
    video_error_code: error.code,
    video_error_message: error.message,
    video_error_recoverable: false,
    video_error_result: 'Error',
    video_error_source: 'Unspecified',
    video_error_value: error.code,
  };

  return {
    name: 'video_error',
    properties,
  };
}
