import type { PlayerAnalyticsEvent } from 'pulsar-utils';
import { logger } from 'tachyon-logger';
import { createPausableInterval } from 'tachyon-utils-stdlib';
import { uniqueIDGenerator } from 'tachyon-utils-twitch';
import type { SpadeBaseStaticProperties } from '../events';
import {
  baseEventProperties,
  bufferEmptyEvent,
  bufferRefillEvent,
  minuteWatchedEvent,
  videoEndEvent,
  videoErrorEvent,
  videoPlayEvent,
} from '../events';
import { getFirstMinuteWatchedDelayMs } from '../utils';

/*
 * https://developer.mozilla.org/en-US/docs/Web/API/HTMLMediaElement/readyState
 */
export enum PlayerReadyState {
  /**
   * No information is available about the media resource.
   */
  HAVE_NOTHING = 0,
  /**
   * Enough of the media resource has been retrieved that the metadata attributes
   * are initialized. Seeking will no longer raise an exception.
   */
  HAVE_METADATA = 1,
  /**
   * Data is available for the current playback position, but not enough to
   * actually play more than one frame.
   */
  HAVE_CURRENT_DATA = 2,
  /**
   * Data for the current playback position as well as for at least a little bit
   * of time into the future is available (in other words, at least two frames of video, for example).
   */
  HAVE_FUTURE_DATA = 3,
  /**
   * Enough data is available—and the download rate is high enough—that the media
   * can be played through to the end without interruption.
   */
  HAVE_ENOUGH_DATA = 4,
}

type AnalyticsArgs = SpadeBaseStaticProperties & {
  onTrackingEvent: (event: PlayerAnalyticsEvent) => void;
  timeSinceLoadStart: number;
  video: HTMLVideoElement;
};

type Buffering = {
  sessionID: string;
  startTimeMS: number;
};

// Temporary note:
// Use https://git.xarth.tv/emerging-platforms/tachyon/tree/69522021cb952e09ce4fca7459a97af9543fb7ae/packages/mobile-web/pulsar-player/src/components/PulsarPlayer
// As implementation reference.

/**
 * Sets up data recording and analytics watching for a given video element.
 * Create this as soon as the video element is available.
 */
export function subscribeVideoAnalytics({
  backend,
  onTrackingEvent,
  timeSinceLoadStart,
  video,
}: AnalyticsArgs): () => void {
  let playSessionId = uniqueIDGenerator();
  let currentVideoSrc = video.src;
  let minutesWatched = 0;
  let firstMinuteWatchedOffsetMs = 0;
  let videoPlayTracked = false;
  let totalSessionBuffers = 0;
  let buffering: Buffering | null = null;

  function getFirstDelayMs() {
    firstMinuteWatchedOffsetMs = getFirstMinuteWatchedDelayMs();
    return firstMinuteWatchedOffsetMs;
  }

  function handleMinuteWatched() {
    minutesWatched += 1;

    onTrackingEvent(
      minuteWatchedEvent({
        firstMinuteWatchedOffsetMs,
        minutesWatched,
        ...baseEventProperties({ backend, playSessionId, video }),
      }),
    );
  }

  const minuteWatchedInterval = createPausableInterval({
    callback: handleMinuteWatched,
    delayMs: 60_000,
    getFirstDelayMs,
    maxDriftBeforeResetMs: 10_000,
  });

  function handleSourceChange() {
    minuteWatchedInterval.stopAndReset();
    playSessionId = uniqueIDGenerator();
    currentVideoSrc = video.src;
    videoPlayTracked = false;
  }

  function handleBufferEmpty() {
    totalSessionBuffers += 1;
    buffering = {
      sessionID: uniqueIDGenerator(),
      startTimeMS: Date.now(),
    };

    onTrackingEvent(
      bufferEmptyEvent({
        bufferEmptyCount: totalSessionBuffers,
        bufferSessionID: buffering.sessionID,
        ...baseEventProperties({ backend, playSessionId, video }),
      }),
    );
  }

  function handleBufferRefill() {
    if (!buffering) {
      logger.error({
        category: 'subscribeVideoAnalytics',
        message: 'Unexpected buffer refill',
        package: 'pulsar-analytics',
      });

      return;
    }

    const { sessionID, startTimeMS } = buffering;
    onTrackingEvent(
      bufferRefillEvent({
        bufferDurationSeconds: (Date.now() - startTimeMS) / 1000,
        bufferEmptyCount: totalSessionBuffers,
        bufferSessionID: sessionID,
        ...baseEventProperties({ backend, playSessionId, video }),
      }),
    );

    buffering = null;
  }

  const playingHandler = () => {
    if (video.src !== currentVideoSrc) {
      handleSourceChange();
    }

    if (!videoPlayTracked) {
      onTrackingEvent(
        videoPlayEvent({
          timeSinceLoadStart,
          ...baseEventProperties({ backend, playSessionId, video }),
        }),
      );
      videoPlayTracked = true;
    }

    minuteWatchedInterval.start();

    if (buffering) {
      handleBufferRefill();
    }
  };
  video.addEventListener('playing', playingHandler);

  const pauseHandler = () => {
    minuteWatchedInterval.pause();
  };
  video.addEventListener('pause', pauseHandler);

  const endedHandler = () => {
    minuteWatchedInterval.pause();
    onTrackingEvent(
      videoEndEvent(baseEventProperties({ backend, playSessionId, video })),
    );
  };
  video.addEventListener('ended', endedHandler);

  const waitingHandler = () => {
    if (!buffering && !video.seeking) {
      handleBufferEmpty();
    }
  };
  video.addEventListener('waiting', waitingHandler);

  const timeUpdateHandler = () => {
    if (buffering && video.readyState === PlayerReadyState.HAVE_ENOUGH_DATA) {
      handleBufferRefill();
    }
  };
  video.addEventListener('timeupdate', timeUpdateHandler);

  const errorHandler = (e: Event) => {
    const videoTarget = e.target as HTMLVideoElement;
    const error = videoTarget.error;
    if (!error) {
      logger.error({
        category: 'subscribeVideoAnalytics',
        message: 'Non-MediaError video error',
        package: 'pulsar-analytics',
      });
    } else {
      onTrackingEvent(
        videoErrorEvent({
          ...baseEventProperties({ backend, playSessionId, video }),
          error,
        }),
      );
      video.pause();
    }
  };
  video.addEventListener('error', errorHandler);

  return () => {
    minuteWatchedInterval.pause();

    video.removeEventListener('playing', playingHandler);
    video.removeEventListener('pause', pauseHandler);
    video.removeEventListener('ended', endedHandler);
    video.removeEventListener('waiting', waitingHandler);
    video.removeEventListener('timeupdate', timeUpdateHandler);
    video.removeEventListener('error', errorHandler);
  };
}
