import EventEmitter from 'eventemitter3';
import { createPausableInterval } from 'tachyon-utils-stdlib';
import { PlayerReadyState, subscribeVideoAnalytics } from '.';

const mockPausableInterval = {
  pause: jest.fn(),
  start: jest.fn(),
  stopAndReset: jest.fn(),
};
jest.mock('tachyon-utils-stdlib', () => ({
  createPausableInterval: jest.fn(() => mockPausableInterval),
}));
const mockCreatePausableInterval = createPausableInterval as jest.Mock;

type MockHTMLVideoElement = HTMLVideoElement & {
  emitEvent: EventEmitter['emit'];
};

function mockVideo(
  overrides: Partial<MockHTMLVideoElement> = {},
): MockHTMLVideoElement {
  const emitter = new EventEmitter();

  return {
    addEventListener: (name: string, handler: VoidFunction) =>
      emitter.addListener(name, handler),
    emitEvent: (...args) => emitter.emit(...args),
    removeEventListener: () => undefined,
    src: 'src1',
    ...overrides,
  } as MockHTMLVideoElement;
}

function setup(videoOverrides: Partial<MockHTMLVideoElement> = {}): {
  onTrackingEvent: jest.Mock;
  tearDown: () => void;
  video: MockHTMLVideoElement;
} {
  const video = mockVideo(videoOverrides);
  const onTrackingEvent = jest.fn();

  return {
    onTrackingEvent,
    tearDown: subscribeVideoAnalytics({
      backend: 'native',
      onTrackingEvent,
      timeSinceLoadStart: 0,
      video,
    }),
    video,
  };
}

describe(subscribeVideoAnalytics, () => {
  describe('video-play', () => {
    it('only emits video-play once per src with a unique play_session_id', () => {
      const { onTrackingEvent, video } = setup();
      expect(onTrackingEvent).not.toHaveBeenCalled();

      video.emitEvent('playing');
      video.emitEvent('playing');
      expect(onTrackingEvent).toHaveBeenCalledTimes(1);
      expect(onTrackingEvent).toHaveBeenCalledWith(
        expect.objectContaining({ name: 'video-play' }),
      );

      video.emitEvent('playing');
      expect(onTrackingEvent).toHaveBeenCalledTimes(1);

      video.src = 'somesrc';
      video.emitEvent('playing');
      expect(onTrackingEvent).toHaveBeenCalledTimes(2);

      // Should change play_session_id when src changes
      expect(onTrackingEvent).toHaveBeenCalledTimes(2);
      expect(
        onTrackingEvent.mock.calls[0][0].properties.play_session_id,
      ).not.toEqual(
        onTrackingEvent.mock.calls[1][0].properties.play_session_id,
      );
    });
  });

  describe('minute-watched', () => {
    it('starts/pauses the interval when expected', () => {
      const { onTrackingEvent, video } = setup();
      expect(mockPausableInterval.start).not.toHaveBeenCalled();

      video.emitEvent('playing');
      expect(mockPausableInterval.start).toHaveBeenCalledTimes(1);

      mockCreatePausableInterval.mock.calls[0][0].callback();
      expect(onTrackingEvent).toHaveBeenCalledWith(
        expect.objectContaining({ name: 'minute-watched' }),
      );

      expect(mockPausableInterval.pause).not.toHaveBeenCalled();
      video.emitEvent('pause');
      expect(mockPausableInterval.pause).toHaveBeenCalledTimes(1);

      video.emitEvent('ended');
      expect(mockPausableInterval.pause).toHaveBeenCalledTimes(2);
    });

    it('pauses the interval on teardown', () => {
      const { tearDown } = setup();

      expect(mockPausableInterval.pause).not.toHaveBeenCalled();
      tearDown();
      expect(mockPausableInterval.pause).toHaveBeenCalledTimes(1);
    });

    it('pauses the old interval and starts a new one on src change', () => {
      const { video } = setup();

      video.src = 'someothersrc';
      video.emitEvent('playing');

      expect(mockPausableInterval.stopAndReset).toHaveBeenCalledTimes(1);
      expect(mockPausableInterval.start).toHaveBeenCalledTimes(1);
    });
  });

  describe('buffer-empty', () => {
    it('emits buffer-empty event when waiting event is emitted, buffering is null, and video is not seeking', () => {
      const { onTrackingEvent, video } = setup();
      video.emitEvent('waiting');
      expect(onTrackingEvent).toHaveBeenCalledTimes(1);
      expect(onTrackingEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          name: 'buffer-empty',
          properties: expect.objectContaining({
            buffer_empty_count: expect.any(Number),
            buffer_session_id: expect.any(String),
          }),
        }),
      );
    });

    it('does not emit buffer-empty event when waiting event is emitted but video is seeking', () => {
      const { onTrackingEvent, video } = setup({ seeking: true });
      video.emitEvent('waiting');
      expect(onTrackingEvent).not.toHaveBeenCalled();
    });
  });

  describe('buffer-refill', () => {
    it('emits buffer-refill event after waiting and playing event are emitted', () => {
      const { onTrackingEvent, video } = setup();
      video.emitEvent('waiting');
      video.emitEvent('playing');
      expect(onTrackingEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          name: 'buffer-refill',
          properties: expect.objectContaining({
            buffer_empty_count: expect.any(Number),
            buffer_session_id: expect.any(String),
            buffering_time: expect.any(Number),
          }),
        }),
      );
    });

    it('emits buffer-refill event after timeupdate event is emitted when the video is buffering and the video ready state is HAVE_ENOUGH_DATA', () => {
      const { onTrackingEvent, video } = setup({
        readyState: PlayerReadyState.HAVE_ENOUGH_DATA,
      });
      video.emitEvent('waiting');
      video.emitEvent('timeupdate');
      expect(onTrackingEvent).toHaveBeenCalledWith(
        expect.objectContaining({ name: 'buffer-refill' }),
      );
    });

    it('does not emit buffer-refill event after timeupdate event is emitted if video ready state is not HAVE_ENOUGH_DATA', () => {
      const { onTrackingEvent, video } = setup({
        readyState: PlayerReadyState.HAVE_NOTHING,
      });
      video.emitEvent('waiting');
      video.emitEvent('timeupdate');
      expect(onTrackingEvent).not.toHaveBeenCalledWith(
        expect.objectContaining({ name: 'buffer-refill' }),
      );
    });
  });

  describe('video_end', () => {
    it('emits video_end event when video ends', () => {
      const { onTrackingEvent, video } = setup();
      video.emitEvent('ended');
      expect(onTrackingEvent).toHaveBeenCalledWith(
        expect.objectContaining({ name: 'video_end' }),
      );
    });
  });

  describe('video_error', () => {
    it('emits video_error event wwhen the video errors', () => {
      const mockVideoPause = jest.fn();
      const { onTrackingEvent, video } = setup({
        error: {
          MEDIA_ERR_ABORTED: 1,
          MEDIA_ERR_DECODE: 3,
          MEDIA_ERR_NETWORK: 2,
          MEDIA_ERR_SRC_NOT_SUPPORTED: 4,
          code: 2,
          message: 'error',
        },
        pause: mockVideoPause,
      });
      video.emitEvent('error', { target: video });
      expect(onTrackingEvent).toHaveBeenCalledWith(
        expect.objectContaining({
          name: 'video_error',
          properties: expect.objectContaining({
            video_error_code: 2,
            video_error_message: 'error',
            video_error_recoverable: false,
            video_error_result: 'Error',
            video_error_source: 'Unspecified',
            video_error_value: 2,
          }),
        }),
      );
      expect(mockVideoPause).toHaveBeenCalledTimes(1);
    });
  });
});
