import { subscribeVideoAnalytics } from 'pulsar-analytics';
import type { PlayerController } from 'pulsar-player-controller';
import { usePlayerControllerSetter } from 'pulsar-player-controller';
import type { PulsarProps } from 'pulsar-utils';
import type { FC } from 'react';
import { useEffect, useRef } from 'react';
import { useConst, useEffectOnce, useLatest } from 'tachyon-utils-react';
import type { Hlsjs } from '../createHlsjsManager';
import { createHlsjsManager } from '../createHlsjsManager';
import { createHlsjsPlayerController } from '../createHlsjsPlayerController';

export type PulsarHlsjsProps = PulsarProps;

// istanbul ignore next: high cost of mocks, low value tests
export const PulsarHlsjs: FC<PulsarHlsjsProps> = ({
  autoPlay,
  src,
  startTime,
  ...props
}) => {
  const setPlayerController = usePlayerControllerSetter();
  const timeSinceLoadStart = useConst(Date.now());

  const videoRef = useRef<HTMLVideoElement | null>(null);
  const playerControllerRef = useRef<PlayerController | null>(null);
  const hlsjsManagerRef = useRef<Hlsjs>();
  // We don't want to reload the player when these values change. These values are only read when the player src changes.
  const initialPlaySettings = useLatest({ autoPlay, startTime });

  useEffectOnce(() => {
    if (videoRef.current) {
      hlsjsManagerRef.current = createHlsjsManager({
        autoPlay: initialPlaySettings.current.autoPlay,
        video: videoRef.current,
      });

      const { controller, emitEvent, tearDown } = createHlsjsPlayerController(
        videoRef.current,
      );
      playerControllerRef.current = controller;

      // subscribeVideoAnalytics will detect video.src changes and reset itself
      const removeVideoAnalytics = subscribeVideoAnalytics({
        backend: 'hlsjs',
        onTrackingEvent: (e) => {
          emitEvent('analytics', e);
        },
        timeSinceLoadStart,
        video: videoRef.current,
      });

      return () => {
        removeVideoAnalytics();
        tearDown();
        hlsjsManagerRef.current?.hlsjsPlayer.destroy();
      };
    }
  });

  useEffect(() => {
    if (playerControllerRef.current) {
      setPlayerController(playerControllerRef.current);
    }
  }, [setPlayerController]);

  useEffect(() => {
    hlsjsManagerRef.current?.setSrc(src, initialPlaySettings.current.startTime);
  }, [src, initialPlaySettings]);

  return (
    <video
      height="100%"
      playsInline
      preload="auto"
      ref={videoRef}
      width="100%"
      {...props}
    />
  );
};

PulsarHlsjs.displayName = 'PulsarHlsjs';
