import { subscribeVideoAnalytics } from 'pulsar-analytics';
import type { PlayerController } from 'pulsar-player-controller';
import { usePlayerControllerSetter } from 'pulsar-player-controller';
import type { PulsarProps } from 'pulsar-utils';
import type { FC } from 'react';
import { useEffect, useRef } from 'react';
import { useConst, useEffectOnce, useLatest } from 'tachyon-utils-react';
import { createNativePlayerController } from '../createNativePlayerController';

export type PulsarNativeProps = PulsarProps;

// istanbul ignore next: high cost of mocks, low value tests
export const PulsarNative: FC<PulsarProps> = ({
  autoPlay,
  src,
  startTime = 0,
  ...props
}) => {
  const setPlayerController = usePlayerControllerSetter();
  const timeSinceLoadStart = useConst(Date.now());

  const videoRef = useRef<HTMLVideoElement | null>(null);
  const playerControllerRef = useRef<PlayerController | null>(null);
  // We don't want to reload the player when these values change. These values are only read when the player src changes.
  const initialPlaySettings = useLatest({ autoPlay, startTime });

  useEffect(() => {
    const video = videoRef.current;
    if (!video) {
      return;
    }

    if (initialPlaySettings.current.startTime) {
      const startTimeHandler = () => {
        video.currentTime = initialPlaySettings.current.startTime;
        video.removeEventListener('loadedmetadata', startTimeHandler);
      };
      video.addEventListener('loadedmetadata', startTimeHandler);
    }

    // Some browsers are less aggressive about attempting playback purely using
    // the "autoPlay" prop on the video element. If autoPlay is desired also
    // attempt it imperatively to increase odds of success.
    if (initialPlaySettings.current.autoPlay) {
      video.play();
    }
  }, [src, initialPlaySettings]);

  useEffectOnce(() => {
    const video = videoRef.current;
    if (!video) {
      return;
    }

    const { controller, emitEvent, tearDown } =
      createNativePlayerController(video);

    playerControllerRef.current = controller;

    const removeVideoAnalytics = subscribeVideoAnalytics({
      backend: 'native',
      onTrackingEvent: (e) => {
        emitEvent('analytics', e);
      },
      timeSinceLoadStart,
      video,
    });

    setPlayerController(controller);

    return () => {
      removeVideoAnalytics();
      tearDown();
    };
  });

  useEffect(() => {
    if (playerControllerRef.current) {
      setPlayerController(playerControllerRef.current);
    }
  }, [setPlayerController]);

  return (
    <video
      height="100%"
      playsInline
      preload="auto"
      ref={videoRef}
      src={src ?? undefined}
      width="100%"
      {...props}
    />
  );
};

PulsarNative.displayName = 'PulsarNative';
