import EventEmitter from 'eventemitter3';
import { createNativePlayerController } from '.';

type MockVideoElement = HTMLVideoElement & {
  emitEvent: (name: string) => void;
};

function mockVideoElement(
  overrides?: Partial<HTMLVideoElement>,
): MockVideoElement {
  const emitter = new EventEmitter();

  const video: Partial<MockVideoElement> = {
    addEventListener: (type: string, listener: any) => {
      emitter.addListener(type, listener);
    },
    emitEvent: (type: string) => {
      emitter.emit(type);
    },
    ...overrides,
  };

  return video as MockVideoElement;
}

describe(createNativePlayerController, () => {
  function setup(videoOverrides?: Partial<HTMLVideoElement>) {
    const video = mockVideoElement(videoOverrides);
    const { controller, emitEvent } = createNativePlayerController(video);

    return {
      controller,
      emitEvent,
      video,
    };
  }

  describe('audio state', () => {
    it('checks muted state on video object', () => {
      const { controller } = setup({
        muted: false,
      });

      expect(controller.isMuted()).toBe(false);
    });
    it('mutes video player', () => {
      const { controller } = setup({
        muted: false,
      });

      expect(controller.isMuted()).toBe(false);

      controller.setMuted(true);

      expect(controller.isMuted()).toBe(true);
    });
    it('checks volume from video element', () => {
      const expectedVolume = 0.5;
      const { controller } = setup({
        volume: expectedVolume,
      });

      expect(controller.getVolume()).toBe(expectedVolume);
    });
    it('sets volume correctly', () => {
      const initialVolume = 0;
      const expectedVolume = 0.5;
      const { controller, video } = setup({
        volume: initialVolume,
      });
      expect(video.volume).toBe(initialVolume);

      controller.setVolume(expectedVolume);
      expect(video.volume).toBe(expectedVolume);
    });
  });

  describe('playback state', () => {
    it('checks if video is paused', () => {
      const { controller } = setup({
        paused: false,
      });

      expect(controller.isPaused()).toBe(false);
    });
    it('pauses video', () => {
      const { controller, video } = setup({
        pause: jest.fn(),
        paused: false,
      });

      controller.pause();
      expect(video.pause).toHaveBeenCalledTimes(1);
    });
    it('plays video', () => {
      const { controller, video } = setup({
        paused: false,
        play: jest.fn(),
      });

      controller.play();
      expect(video.play).toHaveBeenCalledTimes(1);
    });

    it('can play video asynchronously', async () => {
      const { controller, video } = setup({
        paused: false,
        play: jest.fn(),
      });

      const playPromise = controller.play();
      expect(video.play).toHaveBeenCalledTimes(1);
      (video as MockVideoElement).emitEvent('Playing');
      await playPromise;
    });

    it('restarts video playback from the beginning', () => {
      const initialCurrentTime = 100;
      const { controller, video } = setup({
        currentTime: initialCurrentTime,
        play: jest.fn(),
      });

      expect(video.currentTime).toBe(initialCurrentTime);

      controller.restart();

      expect(video.currentTime).toBe(0);
      expect(video.play).toHaveBeenCalledTimes(1);
    });
  });

  describe('sending custom events into the controller', () => {
    it('emits a custom event succesfully to a listener registered through the controller', () => {
      const { controller, emitEvent } = setup();

      const listener = jest.fn();
      controller.subscribeEventListener('analytics', listener);
      emitEvent('analytics', { name: 'some-event', properties: {} });

      expect(listener).toHaveBeenCalledTimes(1);
    });
  });
});
