import {
  createContext,
  useCallback,
  useContext,
  useEffect,
  useMemo,
  useRef,
  useState,
} from 'react';
import { useConstCallback, useUnmount } from 'tachyon-utils-react';
import { usePlayerController } from '../../PlayerControllerRoot';
import type { TextCue } from '../../controller-types';

/**
 * The amount of time to wait before clearing a cue because it no longer applies
 * to the visible content. This is set arbitrarily based on what seems appropriate
 * and also gives the reader enough time to read the cue entirely.
 */
export const CLEAR_LAST_CUE_TIME_MS = 10000;

type SubscriptionHandler = (captions: TextCue | null) => void;
type SubscriptionCanceller = () => void;

export type ClosedCaptionsContext = {
  available: boolean;
  enabled: boolean;
  subscribe: (handler: SubscriptionHandler) => SubscriptionCanceller;
  toggle: () => void;
};

// istanbul ignore next: trivial
export const closedCaptionsContext = createContext<ClosedCaptionsContext>({
  available: false,
  enabled: false,
  subscribe: () => () => undefined,
  toggle: () => undefined,
});

// istanbul ignore next: trivial
/**
 * API for consumers to implement a custom closed caption UI. Use in tandem
 * with ClosedCaptionsRoot.
 */
export function useClosedCaptions(): ClosedCaptionsContext {
  return useContext(closedCaptionsContext);
}

export type UseCaptionManagerProps = {
  /**
   * A unique ID for the content that the captions correspond to.
   * When "contentKey" changes, captions will be cleared.
   */
  contentKey: string;
  defaultEnabled: boolean;
};

/**
 * The captions expectations aren't based on a specific spec. Based on conversation
 * in Jan of 2021 with the Video Playback team, the Closed Captions system should
 * be implemented as follows:
 *
 * 1. A single text cue is emitted per event.
 * 2. A text cue should be eventually cleared to avoid it becoming stale.
 * 3. A text cue will be emitted repeatedly if meant to be shown more than a single second.
 *
 * Test Content:
 * Live: twitch.tv/hungry
 * VOD: twitch.tv/videos/488672793 (Twitch channel TwitchCon archives)
 */
export function useCaptionManager({
  contentKey,
  defaultEnabled = false,
}: UseCaptionManagerProps): ClosedCaptionsContext {
  const [enabled, setEnabled] = useState(defaultEnabled);
  const [available, setAvailable] = useState(false);

  // Each hook registers a text cue listener directly on MediaPlayer rather than
  // using a context to avoid all of the child re-renders from constant
  // provider state updates
  const playerController = usePlayerController();

  const subscriptionIncrementRef = useRef(0);
  const subscriptionsRef = useRef<{ [key: number]: SubscriptionHandler }>({});
  const clearCueTimerRef = useRef<number | null>(null);

  const clearCueRemovalTimer = useConstCallback(() => {
    if (clearCueTimerRef.current !== null) {
      window.clearTimeout(clearCueTimerRef.current);
      clearCueTimerRef.current = null;
    }
  });

  const broadcast = useCallback(
    (cue: TextCue | null) => {
      clearCueRemovalTimer();
      clearCueTimerRef.current = window.setTimeout(() => {
        Object.values(subscriptionsRef.current).forEach((captionSubscriber) => {
          captionSubscriber(null);
        });
      }, CLEAR_LAST_CUE_TIME_MS);

      Object.values(subscriptionsRef.current).forEach((captionSubscriber) => {
        captionSubscriber(cue);
      });
    },
    [clearCueRemovalTimer],
  );

  useEffect(
    () =>
      playerController?.subscribeEventListener('textCue', (cue: TextCue) => {
        setAvailable(true);

        if (enabled) {
          broadcast(cue);
        }
      }),
    [broadcast, enabled, playerController],
  );

  // Reset captions state and clear cue for subscribers when the content changes
  useEffect(() => {
    return () => {
      broadcast(null);
      setAvailable(false);
    };
  }, [broadcast, contentKey]);

  useEffect(() => {
    if (!enabled) {
      // Clear any existing cues if the system was just disabled
      broadcast(null);
    }
  }, [enabled, broadcast]);

  useUnmount(() => {
    clearCueRemovalTimer();
  });

  const toggleCaptions = useConstCallback(() => {
    setEnabled((currentEnabled) => !currentEnabled);
  });

  const subscribe = useConstCallback(
    (subscriptionHandler: SubscriptionHandler) => {
      const subscriptionKey = subscriptionIncrementRef.current;
      subscriptionIncrementRef.current++;

      subscriptionsRef.current[subscriptionKey] = subscriptionHandler;

      return () => {
        delete subscriptionsRef.current[subscriptionKey];
      };
    },
  );

  return useMemo<ClosedCaptionsContext>(
    () => ({
      available,
      enabled,
      subscribe,
      toggle: toggleCaptions,
    }),
    [available, enabled, subscribe, toggleCaptions],
  );
}
