import { act, renderHook } from '@testing-library/react-hooks';
import { usePlayerController } from '../../PlayerControllerRoot';
import type { TextCue } from '../../controller-types';
import type { MockPlayerController } from '../../test-mocks';
import { mockPlayerController } from '../../test-mocks';
import type { UseCaptionManagerProps } from '.';
import { CLEAR_LAST_CUE_TIME_MS, useCaptionManager } from '.';

function mockProps(): UseCaptionManagerProps {
  return {
    contentKey: 'key1',
    defaultEnabled: false,
  };
}

jest.mock('../../PlayerControllerRoot', () => ({
  usePlayerController: jest.fn(),
}));

const mockUsePlayerController = usePlayerController as jest.Mock;

function mockCaption(text: string): TextCue {
  return {
    endTime: 2,
    line: 1,
    position: 20,
    size: 10,
    startTime: 0,
    text,
    type: 'TextCue',
  };
}

describe(useCaptionManager, () => {
  let mockedMediaPlayer: MockPlayerController;

  beforeEach(() => {
    mockedMediaPlayer = mockPlayerController();
    mockUsePlayerController.mockReturnValue(mockedMediaPlayer);
  });

  it('updates state to available when a text cue is received', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: mockProps(),
    });

    expect(result.current.available).toEqual(false);

    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line'));
    });
    expect(result.current.available).toEqual(true);
  });

  it('toggles through enabled/disabled successfully', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: mockProps(),
    });
    expect(result.current.enabled).toEqual(false);

    act(() => {
      result.current.toggle();
    });
    expect(result.current.enabled).toEqual(true);

    act(() => {
      result.current.toggle();
    });
    expect(result.current.enabled).toEqual(false);
  });

  it('does not invoke caption subscribers if disabled', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: mockProps(),
    });

    const mockCaptionHandler = jest.fn();
    result.current.subscribe(mockCaptionHandler);

    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line'));
    });
    expect(mockCaptionHandler).not.toHaveBeenCalled();
  });

  it('emits the cue to subscribers when the captions system is enabled', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: { contentKey: 'therealderekt', defaultEnabled: true },
    });

    const mockCaptionHandler = jest.fn();
    result.current.subscribe(mockCaptionHandler);
    expect(mockCaptionHandler).not.toHaveBeenCalled();

    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line'));
    });
    expect(mockCaptionHandler).toHaveBeenLastCalledWith(
      expect.objectContaining({ text: 'caption line' }),
    );

    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line 2'));
    });
    expect(mockCaptionHandler).toHaveBeenLastCalledWith(
      expect.objectContaining({ text: 'caption line 2' }),
    );

    const captionSet2 = mockCaptionHandler.mock.calls[1][0];
    expect(captionSet2.text).toEqual('caption line 2');

    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line 3'));
    });

    expect(mockCaptionHandler).toHaveBeenCalledTimes(3);
    expect(mockCaptionHandler).toHaveBeenLastCalledWith(
      expect.objectContaining({ text: 'caption line 3' }),
    );
  });

  it('clears a cue when a replacement cue is not emitted within a second', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: { contentKey: 'therealderekt', defaultEnabled: true },
    });

    const mockCaptionHandler = jest.fn();
    result.current.subscribe(mockCaptionHandler);
    expect(mockCaptionHandler).not.toHaveBeenCalled();

    // trivial expiry case
    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line'));
    });
    expect(mockCaptionHandler).toHaveBeenCalledTimes(1);

    jest.advanceTimersByTime(CLEAR_LAST_CUE_TIME_MS - 1);
    expect(mockCaptionHandler).toHaveBeenCalledTimes(1);

    jest.advanceTimersByTime(1);
    expect(mockCaptionHandler).toHaveBeenCalledTimes(2);
    expect(mockCaptionHandler).toHaveBeenLastCalledWith(null);
  });

  it('clears a cue when captions are toggled off', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: { contentKey: 'therealderekt', defaultEnabled: true },
    });

    const mockCaptionHandler = jest.fn();
    result.current.subscribe(mockCaptionHandler);
    expect(mockCaptionHandler).not.toHaveBeenCalled();

    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line'));
    });
    expect(mockCaptionHandler).toHaveBeenCalledTimes(1);

    act(() => {
      result.current.toggle();
    });
    expect(mockCaptionHandler).toHaveBeenCalledTimes(2);
    expect(mockCaptionHandler).toHaveBeenLastCalledWith(null);
  });

  it('resets the clear timer when another cue is emitted within a second', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: { contentKey: 'therealderekt', defaultEnabled: true },
    });

    const mockCaptionHandler = jest.fn();
    result.current.subscribe(mockCaptionHandler);
    expect(mockCaptionHandler).not.toHaveBeenCalled();

    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line'));
    });
    expect(mockCaptionHandler).toHaveBeenCalledTimes(1);

    jest.advanceTimersByTime(CLEAR_LAST_CUE_TIME_MS - 1);
    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line 2'));
    });
    expect(mockCaptionHandler).toHaveBeenCalledTimes(2);

    jest.advanceTimersByTime(1);
    expect(mockCaptionHandler).toHaveBeenCalledTimes(2);
    jest.advanceTimersByTime(CLEAR_LAST_CUE_TIME_MS - 1);

    expect(mockCaptionHandler).toHaveBeenCalledTimes(3);
    expect(mockCaptionHandler).toHaveBeenLastCalledWith(null);
  });

  it('resets state when the content key changes', () => {
    const { rerender, result } = renderHook((p) => useCaptionManager(p), {
      initialProps: mockProps(),
    });
    const mockCaptionHandler = jest.fn();
    result.current.subscribe(mockCaptionHandler);

    act(() => {
      result.current.toggle();
    });
    act(() => {
      mockedMediaPlayer.emitEvent('textCue', mockCaption('caption line'));
    });
    expect(result.current.available).toEqual(true);
    expect(result.current.enabled).toEqual(true);
    expect(mockCaptionHandler).toHaveBeenCalledTimes(1);

    rerender({ ...mockProps(), contentKey: 'otherKey' });
    expect(result.current.available).toEqual(false);
    expect(result.current.enabled).toEqual(true);

    expect(mockCaptionHandler).toHaveBeenCalledTimes(2);
    expect(mockCaptionHandler).toHaveBeenLastCalledWith(null);
  });

  it('initiates with captions enabled when specified', () => {
    const { result } = renderHook((p) => useCaptionManager(p), {
      initialProps: { contentKey: '1', defaultEnabled: true },
    });

    expect(result.current.enabled).toEqual(true);
  });
});
