import type { Dispatch, FC } from 'react';
import { createContext, useContext, useState } from 'react';
import { useUnmount } from 'tachyon-utils-react';
import type { PlayerController } from '../controller-types';

type PlayerControllerSetterContext<QualityExtension = {}> = [
  PlayerController<QualityExtension> | null,
  Dispatch<PlayerController<QualityExtension> | null>,
];

/**
 * This setter context is intentionally separate from playerControllerContext to avoid double rendering
 * that could otherwise occur as a result of a new context object being calculated when a controller is
 * set for the first time.
 */
const playerControllerSetterContext = createContext<
  PlayerControllerSetterContext<any>
>([null, () => undefined]);

/**
 * Used internally by Pulsar backends to interface with the player controller system.
 */
export function usePlayerControllerSetter<QualityExtension = {}>(): (
  controller: PlayerController<QualityExtension>,
) => void {
  const [, setPlayerController] = useContext<
    PlayerControllerSetterContext<QualityExtension>
  >(playerControllerSetterContext);

  useUnmount(() => {
    setPlayerController(null);
  });

  return setPlayerController;
}

type PlayerControllerContext = PlayerController | null;
const playerControllerContext = createContext<PlayerControllerContext>(null);

// istanbul ignore next: trivial
/**
 * Allows a consumer to efficiently update its state when the player it is
 * trying to control becomes available or unavailable, without unnecessarily
 * causing re-renders if the underlying controller updates.
 */
export function usePlayerController(): PlayerControllerContext {
  return useContext(playerControllerContext);
}

/**
 * Provides the ability to get and set an active Player controller via usePlayerController for
 * features that need to interact imperatively with the underlying video player's state.
 */
export const PlayerControllerRoot: FC = ({ children }) => {
  const playerControllerState = useState<PlayerController | null>(null);
  const [playerController] = playerControllerState;

  return (
    <playerControllerSetterContext.Provider value={playerControllerState}>
      <playerControllerContext.Provider value={playerController}>
        {children}
      </playerControllerContext.Provider>
    </playerControllerSetterContext.Provider>
  );
};

PlayerControllerRoot.displayName = 'PlayerControllerRoot';
