import type EventEmitter from 'eventemitter3';
import type { Quality } from 'pulsar-utils';
import type { ControllerEventMap, PlayerEventHandler } from './event';
import type { PlaybackState } from './playbackState';

export type ControllerEventEmitter = EventEmitter<PlayerControllerEvent>;

type SyntheticPlayerEvents = Pick<ControllerEventMap, 'error'>;
export type PlayerControllerEvent = CustomEmitterEvents & SyntheticPlayerEvents;

export type CreatePlayerController = {
  controller: PlayerController;
  /**
   * Allows the player controller creator to augment the base event system.
   */
  emitEvent: <K extends keyof PlayerControllerEvent>(
    type: K,
    event: PlayerControllerEvent[K],
  ) => void;
  /**
   * Must be invoked before discarding the PlayerController instance to ensure
   * that listeners and other state subscribers are cleaned up.
   */
  tearDown: VoidFunction;
};

/**
 * The list of events that are "custom" and do not map to the standard
 * events emitted by an HTMLVideoElement.
 */
export type CustomEmitterEvents = Pick<
  ControllerEventMap,
  | 'adCue'
  | 'analytics'
  | 'audioBlocked'
  | 'playbackBlocked'
  | 'playbackStateChange'
  | 'qualityChange'
  | 'streamSourceCue'
  | 'textCue'
>;

export type QualityManager<Extensions = {}> = {
  getQualities: () => Quality<Extensions>[];
  getQuality: () => Quality<Extensions>;
  isAutoQualityMode: () => boolean;
  setAutoQualityMode: (auto: boolean) => void;
  setQuality: (quality: Quality<Extensions>) => void;
};

export type EventListenerManager = {
  /**
   * @return {function} invoke the returned callback to unsubscribe the event listener
   */
  subscribeEventListener<K extends keyof ControllerEventMap>(
    name: K,
    handler: PlayerEventHandler<K>,
  ): () => void;
};

export type PlaybackManager = {
  /**
   * @return {float}
   */
  getDuration: () => number;
  /**
   * @return {float}
   */
  getPosition: () => number;
  /**
   * @return {float}
   */
  getVolume: () => number;
  isMuted: () => boolean;
  isPaused: () => boolean;
  pause: () => void;
  play: () => Promise<void>;
  restart: () => void;
  /**
   * @param {float} position
   */
  seekTo: (position: number) => void;
  setMuted: (muted: boolean) => void;
  /**
   * @param {float} volume
   */
  setVolume: (volume: number) => void;
};

export type PlayerController<QualityExtensions = {}> = EventListenerManager &
  PlaybackManager &
  QualityManager<QualityExtensions> & {
    getPlaybackState: () => PlaybackState;
  };
