import type { PlayerAnalyticsEvent } from 'pulsar-utils';
import { browserSupportsHLS } from 'pulsar-utils';
import { exhaustedCase } from 'tachyon-utils-ts';
import type {
  ControllerEventEmitter,
  ControllerEventMap,
  EventListenerManager,
  PlayerError,
  PlayerEventHandler,
} from '../../controller-types';
import { ErrorType } from '../../controller-types';

// https://developer.mozilla.org/en-US/docs/Web/API/MediaError/code
export const mediaErrorCode = {
  MEDIA_ERR_ABORTED: 1,
  MEDIA_ERR_DECODE: 3,
  MEDIA_ERR_NETWORK: 2,
  MEDIA_ERR_SRC_NOT_SUPPORTED: 4,
};

export function parseNativeVideoError(video: HTMLVideoElement): PlayerError {
  const { error, src } = video;
  const isHlsSrc = src.indexOf('.m3u8') !== -1;
  const source = isHlsSrc ? 'MasterPlaylist' : 'native';

  if (!browserSupportsHLS(video)) {
    return {
      code: -1,
      message: 'browser does not support HLS',
      source: 'native',
      type: ErrorType.NOT_SUPPORTED,
    };
  }

  // If we attempt to play an offline HLS stream the video errors with a NOT_SUPPORTED (code 4), so we reclassify it and raise ERROR_NOT_AVAILABLE instead
  if (error?.code === mediaErrorCode.MEDIA_ERR_SRC_NOT_SUPPORTED && isHlsSrc) {
    return {
      code: 404,
      message: error?.message ?? '',
      source,
      type: ErrorType.NOT_AVAILABLE,
    };
  }

  return {
    code: error?.code ?? -1,
    message: error?.message ?? '',
    source,
    type: ErrorType.GENERIC,
  };
}

// istanbul ignore next: trivial
export function createNativeEventListenerManager(
  video: HTMLVideoElement,
  eventEmitter: ControllerEventEmitter,
): EventListenerManager {
  return {
    subscribeEventListener<K extends keyof ControllerEventMap>(
      name: K,
      handler: PlayerEventHandler<K>,
    ): VoidFunction {
      switch (name) {
        case 'error': {
          const cb = (err: PlayerError) => {
            (handler as PlayerEventHandler<'error'>)(err);
          };
          eventEmitter.addListener('error', cb);

          // emit into event emitter system to make sure any other registered error callbacks are respected
          const cb2 = () => {
            eventEmitter.emit('error', parseNativeVideoError(video));
          };
          video.addEventListener('error', cb2);

          return () => {
            video.removeEventListener('error', cb2);
            eventEmitter.removeListener('error', cb);
          };
        }
        case 'playing': {
          const cb = () => {
            (handler as PlayerEventHandler<'playing'>)();
          };
          video.addEventListener('playing', cb);
          return () => video.removeEventListener('playing', cb);
        }
        case 'timeUpdate': {
          const cb = () => {
            (handler as PlayerEventHandler<'timeUpdate'>)();
          };
          video.addEventListener('timeupdate', cb);
          return () => video.removeEventListener('timeupdate', cb);
        }
        case 'volume': {
          const cb = () => {
            (handler as PlayerEventHandler<'volume'>)();
          };
          video.addEventListener('volumechange', cb);
          return () => video.removeEventListener('volumechange', cb);
        }
        /**
         * Events derived from custom sources.
         */
        case 'analytics':
          const cb = (event: PlayerAnalyticsEvent) => {
            (handler as PlayerEventHandler<'analytics'>)(event);
          };
          eventEmitter.addListener('analytics', cb);
          return () => eventEmitter.removeListener('analytics', cb);
        case 'playbackStateChange': {
          eventEmitter.addListener('playbackStateChange', handler);
          return () => {
            eventEmitter.removeListener('playbackStateChange', handler);
          };
        }
        /**
         * No-op. Not supported for Native HLS playback.
         */
        case 'adCue':
        case 'audioBlocked':
        case 'playbackBlocked':
        case 'qualityChange':
        case 'textCue':
          return () => undefined;
        default:
          return exhaustedCase(name as never, () => undefined);
      }
    },
  };
}
