import { browserSupportsHLS } from 'pulsar-utils';
import { ErrorType } from '../..';
import { mediaErrorCode, parseNativeVideoError } from '.';

jest.mock('pulsar-utils', () => ({
  browserSupportsHLS: jest.fn(() => true),
}));
const mockBrowserSupportsHLS = browserSupportsHLS as jest.Mock;

function mockVideoElement(
  overrides?: Partial<HTMLVideoElement>,
): HTMLVideoElement {
  return {
    error: {
      code: 1,
      message: '',
    },
    src: 'toothepaste.m3u8',
    ...overrides,
  } as HTMLVideoElement;
}

describe(parseNativeVideoError, () => {
  describe('NOT_SUPPORTED Error cases', () => {
    it('browser does not support HLS', () => {
      const video = mockVideoElement();
      mockBrowserSupportsHLS.mockReturnValueOnce(false);

      expect(parseNativeVideoError(video)).toEqual({
        code: -1,
        message: 'browser does not support HLS',
        source: 'native',
        type: ErrorType.NOT_SUPPORTED,
      });
    });
  });
  describe('NOT_AVAILABLE Error cases', () => {
    it('offline channel', () => {
      const video = mockVideoElement({
        error: {
          code: mediaErrorCode.MEDIA_ERR_SRC_NOT_SUPPORTED,
          message: '',
        } as MediaError,
      });

      expect(parseNativeVideoError(video)).toEqual({
        code: 404,
        message: '',
        source: 'MasterPlaylist',
        type: ErrorType.NOT_AVAILABLE,
      });
    });
  });

  describe('GENERIC Error cases', () => {
    it('non-offline channel cases', () => {
      const code = mediaErrorCode.MEDIA_ERR_ABORTED;
      const message = '';

      const video = mockVideoElement({
        error: {
          code,
          message,
        } as MediaError,
      });

      expect(parseNativeVideoError(video)).toEqual({
        code,
        message,
        source: 'MasterPlaylist',
        type: ErrorType.GENERIC,
      });
    });

    it('MediaError.code is MEDIA_ERR_SRC_NOT_SUPPORTED, but video.src is not an m3u8 extension', () => {
      const code = mediaErrorCode.MEDIA_ERR_SRC_NOT_SUPPORTED;
      const message = '';
      const video = mockVideoElement({
        error: {
          code,
          message,
        } as MediaError,
        src: 'potato.mp4',
      });

      expect(parseNativeVideoError(video)).toEqual({
        code,
        message,
        source: 'native',
        type: ErrorType.GENERIC,
      });
    });
  });
});
